"""module defining extracting functions"""

from functools import partial

from nile.api.v1 import (
    Record,
    aggregators as na,
    extractors as ne,
)
from qb2.api.v1 import (
    filters as qf
)

from geo_utils import get_dist_in_km


def get_permalink(v):
    if isinstance(v, dict) and v.get('permalink'):
        return int(v.get('permalink'))
    return None


def get_name(names):
    for name in names:
        if name['type'] == 'main' and name['value']['locale'] == 'ru':
            return name['value']['value']
    return None


def get_coordinates(field):
    if field and field.get('pos'):
        return field.get('pos').get('coordinates')
    return None


def get_points(p, t, v):
    if p == 'maps_www.panorama_player' and t == 'change_state' and isinstance(v, dict):
        return v.get('point')
    return None


def recalculate_permalinks(job, permalinks):
    permalinks_info = job.table(
        '$altay'
    ).filter(
        qf.custom(lambda x: x in permalinks, 'permalink')
    ).project(
        'permalink',
        coordinates=ne.custom(lambda x: get_coordinates(x), 'address'),
        name=ne.custom(lambda x: get_name(x), 'names')
    )
    return permalinks_info


def is_near_4neq(dict_of_points, pp, R=.2):
    min_dist = 10 ** 10
    if isinstance(dict_of_points, dict):
        for key, val in dict_of_points.items():
            min_dist = min(min_dist, get_dist_in_km(tuple(val), pp))
            if min_dist < R:
                return key
    return None


opened_cards_paths = [
    'maps_www.poi_panel.preview_card',
    'maps_www.serp_panel.preview_card',
    'maps_www.orgpage.content'
]

opened_panorama_paths = [
    'maps_www.serp_panel.preview_card.panorama',
    'maps_www.poi_panel.preview_card.panorama',
    'maps_www.orgpage.content.contacts.address.panorama'
]
panoramas_movings = ['maps_www.panorama_player']


def panoramas_metric_reducer(groups, permalinks_points):
    for key, records in groups:
        for record in records:
            path = record.path
            event_type = record.event_type

            cards_opening = int(path in opened_cards_paths and event_type == 'show')
            panoramas_opening = int(path in opened_panorama_paths and event_type == 'click')
            permalink = get_permalink(record.vars)

            panoramas_points = get_points(record.path, record.event_type, record.vars)
            panoramas_player_opening = int(panoramas_points is not None)
            if not permalink and panoramas_points:
                permalink = is_near_4neq(permalinks_points, panoramas_points)

            actual_action = cards_opening + panoramas_opening + panoramas_player_opening
            permalinks = permalinks_points.keys()
            if permalink in permalinks and actual_action > 0:
                yield Record(
                    key,
                    cards_opening=cards_opening,
                    panoramas_opening=panoramas_opening,
                    permalink=permalink,
                    panoramas_points=panoramas_points,
                    panoramas_player_opening=panoramas_player_opening
                )


def metrics_by_sessions_reduce(groups):
    for key, records in groups:
        # returns dict {field_name: sum_of_values, ..}
        total_by_key = records.sum_fields(
            'cards_opening',
            'panoramas_opening',
            'panoramas_player_opening'
        )
        yield Record(
            key,
            cards_opening=int(total_by_key.cards_opening > 0),
            panoramas_opening=int(total_by_key.panoramas_opening > 0),
            panoramas_player_opening=int(total_by_key.panoramas_player_opening > 0)
        )


def extract_panoramas_metrics(job, permalinks_points):
    redir_log = job.table(
        '$redir/@dates'
    ).qb2(
        intensity='data',
        log='redir-log-bebr',
        mode='yamr_lines',
        fields=[
            'vars',
            'path',
            'datetime',
            'event_type',
            'uid',
            'session_id',
            'geo_id'
        ],
        filters=[
            qf.default_filtering('redir-log-bebr'),
            qf.equals('pid', '30'),

        ]
    )

    opened_cards_count = redir_log.groupby(
        'session_id',
        'uid'
    ).reduce(
        partial(
            panoramas_metric_reducer,
            permalinks_points=permalinks_points
        )
    ).groupby(
        'session_id',
        'uid',
        'permalink'
    ).reduce(
        metrics_by_sessions_reduce
    ).groupby(
        'uid',
        'permalink'
    ).aggregate(
        count_cards_opening=na.sum('cards_opening'),
        count_panoramas_opening=na.sum('panoramas_opening'),
        count_panoramas_player_opening=na.sum('panoramas_player_opening')
    ).project(
        ne.all(),
        count_cards_opening_by_user=ne.custom(
            lambda x: int(x > 0), 'count_cards_opening'
        ),
        count_panoramas_opening_by_user=ne.custom(
            lambda x: int(x > 0), 'count_panoramas_opening'
        ),
        count_panoramas_player_opening_by_user=ne.custom(
            lambda x: int(x > 0), 'count_panoramas_player_opening'
        )
    ).groupby(
        'permalink'
    ).aggregate(
        sessions_with_opening_cards=na.sum('count_cards_opening'),
        sessions_with_opening_panoramas=na.sum('count_panoramas_opening'),
        sessions_with_moving_in_player=na.sum('count_panoramas_player_opening'),

        users_with_opening_cards=na.sum('count_cards_opening_by_user'),
        users_with_opening_panoramas=na.sum('count_panoramas_opening_by_user'),
        users_with_moving_in_player=na.sum('count_panoramas_player_opening_by_user')
    )

    return opened_cards_count
