import argparse
import logging
from datetime import datetime, timedelta

from dateutil.relativedelta import relativedelta
from nile.api.v1 import (
    clusters
)

from extractors import extract_panoramas_metrics, recalculate_permalinks


def get_dates_interval(start_date, finish_date):
    return '{{{}..{}}}'.format(start_date, finish_date)


def calculating_panoramas_metric(
        start_date,
        finish_date,
        path,
        update_permalinks_info):
    """prepare and run job"""

    cluster = clusters.yt.Hahn().env(
        templates={
            'altay': '/home/sprav/altay/prod/snapshot/company',
            'redir': '/logs/redir-log/1d',
            'task_root': path
        },
        yt_spec_defaults=dict(
            pool_trees=["physical"],
            tentative_pool_trees=["cloud"]
        )
    )

    job = cluster.job('calculate-4neq-panoramas').env(
        files=['extractors.py', 'geo_utils.py'],
        templates={
            'dates': get_dates_interval(start_date, finish_date)
        }
    )

    permalinks = set(rec.permalink for rec in cluster.read(
        '/home/geo-analytics/denislina/193panoramas/address_permalink'
    ))

    if update_permalinks_info:
        job_for_updates = cluster.job('update-panoramas-info').env(
            files=['extractors.py', 'geo_utils.py']
        )
        permalinks_info = recalculate_permalinks(job_for_updates, permalinks)
        permalinks_info.put(
            '$task_root/permalinks_info'
        )
        job_for_updates.run()

    permalinks_points = dict(
        [rec.permalink, rec.coordinates] for rec in cluster.read('$task_root/permalinks_info')
    )

    opened_cards_count = extract_panoramas_metrics(job, permalinks_points)
    opened_cards_count.put(
        '$task_root/before_join_{}..{}'.format(start_date, finish_date)
    )

    permalinks_info = job.table('$task_root/permalinks_info')
    result = opened_cards_count.join(
        permalinks_info,
        by='permalink',
        type='right'
    )
    result.put(
        '$task_root/{}..{}'.format(start_date, finish_date)
    )

    job.run()


def valid_date(date):
    try:
        datetime.strptime(date, "%Y-%m-%d")
        return date
    except ValueError:
        msg = "Not a valid date: '{}'.".format(date)
        raise argparse.ArgumentTypeError(msg)


def str2bool(value):
    if value.lower() in ('yes', 'true', 't', 'y', '1'):
        return True
    elif value.lower() in ('no', 'false', 'f', 'n', '0'):
        return False
    else:
        raise argparse.ArgumentTypeError('Boolean value expected')


def checking_dates_period(start_date, finish_date):
    yesterday = str((datetime.now() - timedelta(days=1)).date)
    logger = logging.getLogger('extract')
    if start_date > finish_date:
        logger.error('start date more than finish date')
    if finish_date > yesterday:
        logger.error('finish date more than yesterday')


if __name__ == '__main__':
    parser = argparse.ArgumentParser()
    parser.add_argument(
        '-s',
        '-start_date',
        type=valid_date,
        required=True,
        help='the start date for report - format YYYY-MM-DD'
    )
    parser.add_argument(
        '-f',
        '-finish_date',
        type=valid_date,
        required=False,
        help='the finish date for report - format YYYY-MM-DD'
    )

    parser.add_argument(
        '-u',
        '-update_table',
        type=str2bool,
        default=True,
        help='flag for calculating panorama players clicks'
    )

    parser.add_argument(
        '-p',
        '-path',
        type=str,
        default='home/geo-analytics/denislina/193-calculate-panoramas',
        help='path to result report'
    )
    args = parser.parse_args()
    if not args.f:
        args.f = args.s + relativedelta(months=1, days=-1)

    checking_dates_period(start_date=args.s,
                          finish_date=args.f)

    calculating_panoramas_metric(start_date=args.s,
                                 finish_date=args.f,
                                 path=args.p,
                                 update_permalinks_info=args.u)
