#!/usr/bin/env python
#  -*- coding: utf-8 -*-
'''module defining extracting functions for the
collections cube calculations
'''

from qb2.api.v1 import filters as qf, extractors as qe
from nile.api.v1 import Template, extractors as ne


def find_earliest_squize(cluster, dates):
    '''read pathes, find earliest squize'''
    old_date = ''
    daterange = Template(dates).unfold(stringify=True)

    for date in daterange:
        try:
            cluster.read('$bebr_squized/%s' % date)

            return date, old_date
        except ValueError:
            old_date = date


def make_bebr_stream(table, redir=True):
    '''returns bebr stream — either from redir or from a squize'''
    stream = table.qb2(
        intensity='data',
        log='redir-log-bebr',
        mode='yamr_lines' if redir else 'records',
        fields=[
            'session_id', 'yandexuid', 'path', 'event_type', 'event_index',
            'vars', 'date'
        ],
        filters=[
            qf.equals('dtype', 'bebr'),
            qf.defined('date', 'yandexuid'),
            qf.equals('pid', '30'),
            qf.default_filtering('redir-log-bebr'),
            qf.equals('path', 'maps_www.discovery'),
            qf.equals('event_type', 'show')
        ]
    ).unique('event_index', 'session_id')

    return stream


def stream_desktop(cluster, job):
    '''returns bebr stream for join on collections'''
    dates = cluster.environment.templates['dates']
    earliest_squize, last_redir = find_earliest_squize(cluster, dates)
    assert earliest_squize is not None, 'no squizes found'
    date_from, date_to = dates[1:-1].split('..')
    redir_dates = '{%s}' % '..'.join([date_from, last_redir])
    squize_dates = '{%s}' % '..'.join([earliest_squize, date_to])
    print 'squize_dates: %s' % squize_dates

    squize_tables = job.table(
        '$bebr_squized/%s' % squize_dates, ignore_missing=True
    )
    squize_stream = make_bebr_stream(squize_tables, False)

    if last_redir:
        print 'redir dates: %s' % redir_dates
        redir_tables = job.table(
            '$redir_log/%s' % redir_dates, ignore_missing=True
        )
        redir_stream = make_bebr_stream(redir_tables)
        squize_redir_stream = job.concat(redir_stream, squize_stream)

        return squize_redir_stream

    return squize_stream


def stream_mobile(job):
    '''prepare and return a mobile app stream for collections join'''
    stream = job.table(
        '$mobmetrika/@dates', ignore_missing=True
    ).qb2(
        intensity='data',
        log='metrika-mobile-log',
        fields=[
            'event_date',
            'event_datetime',
            'event_number',
            'event_type',
            'event_name',
            'app_platform',
            'app_version',
            qe.mobile.event_value('event_value'),
            'device_id',
            'session_type',
        ],
        filters=[
            qf.defined('event_date'),
            qf.apps_projects('Mobile_Soft_Maps'),
            qf.equals('event_name', 'discovery.opened'),
            qf.equals('session_type', 'SESSION_FOREGROUND'),
        ],
    )

    return stream


def extract(cluster, job):
    '''returns basic streams for historical collections cube processing'''
    desktop_stream = stream_desktop(cluster, job)
    mobile_stream = stream_mobile(job)

    collections_stream = job.table(
        '$collections_history/@dates', ignore_missing=True
    ).project(
        ne.all(),
        qe.table_path('table_path').hide(),
        fielddate=ne.custom(lambda x: x.split('/')[-1], 'table_path')
    )

    return (collections_stream, desktop_stream, mobile_stream)
