import sys
import csv
import json
from collections import OrderedDict
from nile.api.v1 import clusters, Record
from qb2.api.v1 import filters as qf
from functools import reduce
import argparse


def get_icon_class(icon_class):
    if icon_class is None:
        return
    else:
        return icon_class

def main(cluster, rank_path, icon_info_filter_path, geojson_out_path):
    rank = cluster.read(rank_path).as_dataframe().set_index('permalink')
    icon_info_filter = cluster.read(icon_info_filter_path).as_dataframe().set_index('permalink')
    df_list = [rank, icon_info_filter]
    stream = reduce(lambda x, y: x.join(y, how='inner'), df_list)

    records = [Record(
                      permalink=index, 
                      min_zoom=row['min_zoom'], 
                      geo_id=row['geo_id'], 
                      is_geoproduct=row['is_geoproduct'], 
                      lat=row['lat'], 
                      lon=row['lon'],
                      name=row['name'],
                      disp_class_tweak=row['disp_class_tweak'], 
                      disp_class=row['disp_class'], 
                      icon_class=row['icon_class']
                     ) for index, row in stream.iterrows()]
    cluster.write(geojson_out_path, records)

    features = []
    for idx, record in stream.iterrows():
        feature = dict(
                type='Feature',
                geometry=dict(
                    type='Point',
                    coordinates=(record.lon, record.lat)
                ),
                properties=dict(
                    name=record['name'],
                    permalink=idx,
                    gp=str(record.is_geoproduct).lower(),
                    icon_class=get_icon_class(record.icon_class),
                    zmin=record.min_zoom,
                    zmax=21,
                ),
            )
        features.append(feature)
    obj = dict(
        type='FeatureCollection',
        features=features,
    )
    with open('geo.json', 'w') as f:
        json.dump(obj, f, ensure_ascii=False)
    
if __name__ == '__main__':
    
    cluster = clusters.Hahn()
    
    parser = argparse.ArgumentParser()
    parser.add_argument('-v', required=True, help='version of ranking algorithm')
    args = parser.parse_args()

    geojson_out_path = '//home/geo-analytics/poi_rank/' + args.v + '/filtered/moscow/geojson'
    rank_path = '//home/geo-analytics/poi_rank/' + args.v + '/filtered/moscow/rank'
    icon_info_filter_path = '//home/geo-analytics/poi_rank/' + args.v + '/filtered/moscow/icon_info'

    main(cluster, rank_path, icon_info_filter_path, geojson_out_path)
