# coding=utf-8
"""
Определения кликовых метрик в картах и навигаторе
"""
import json
import re
from library.python import resource
import cyson as yson

GOAL_DEEP_USE_ALIASES = {
    "build_route",
    "show_phone",
    "make_call",
    "geoproduct_cta",
    "open_panorama",
    # "any_site",
    "show_entrances",
    "cta",
    "bookmark",
    "sharing",
    "call_taxi",
    "last_mile_story"
}


def load_clicks_description(filename="clicks_description.json"):
    arcadia_resource = resource.find("/analytics/geo/maps/common/clicks_description_json")
    if arcadia_resource is not None:
        clicks_description = json.loads(arcadia_resource.decode("utf-8"))
    else:
        with open(filename, "r") as f:
            clicks_description = json.load(f)
    for service_data in clicks_description:
        for click_info in service_data["clicks"]:
            for event in click_info["events"]:
                if "regexp" in event:
                    event["regexp"] = re.compile(event["regexp"])
    return clicks_description


def match_event(event_name, event_params, condition):
    if "regexp" in condition:
        name_match = condition["regexp"].match(event_name) is not None
    else:
        name_match = condition["name"] == event_name
    if name_match:
        condition_params = condition.get("params", {})
        return all(str((event_params or {}).get(param)) == str(value) for param, value in condition_params.items())


def get_click_parser(filename="clicks_description.json"):
    """
    Метод используется в расчетах YQL https://yql.yandex-team.ru/Operations/X9uTCCyLNSobPDm-ZKlTatKZ7ns4xhJniC7ib8O9mKc=
    """
    clicks_description = load_clicks_description(filename)
    clicks_description = dict((service_data["service"], service_data["clicks"]) for service_data in clicks_description)

    # https://st.yandex-team.ru/MAPSPRODUCT-336
    # В Нави встроена поисковая библиотека, которая создана на основе логики и UI МЯКа.
    # С библиотекой в Нави приехало и логгирование поискового сценария как в МЯКе.
    # Поэтому события с карточкой места в новой версии Нави буду как в МЯКе.
    # Для обработки всех версий Нави необходимо использовать и события из navi и из mobile_maps",
    clicks_description['navi'].extend(clicks_description['mobile_maps'])

    def parse_click(service, event_name, event_params):
        clicks_info = clicks_description.get(service, [])
        aliases = set()
        if not event_name:
            return aliases
        for click in clicks_info:
            for event in click["events"]:
                if match_event(event_name, event_params, event):
                    aliases.add(click["alias"])
        return list(aliases)

    return parse_click


parse_click = get_click_parser()


def is_good_use_mobile_maps_search_lib(click):
    return click.get("path") == "search.show-place-card" or \
        click.get("path") == "search.open-place-view" and click.get("source") in ("serp", "search-snippet")


def is_good_use(client, click):
    try:
        click = yson.loads((click or {}))
    except:
        pass
    if not bool(click.get("path")):
        return False
    if client == "maps":
        return click.get("path") in ("maps_www.map.search_results.placemark", "maps_www.serp_panel.results.result_item")
    elif client == "mobile_maps":
        return is_good_use_mobile_maps_search_lib(click)
    elif client == "navi":
        return click.get("path") == "map.show-minicard" or is_good_use_mobile_maps_search_lib(click)
    else:
        return False

WEB_MAPS_ROUTES_DEEPUSE_PREFIXES = [
    "maps_www.routes_panel.results.filter_settings",
    "maps_www.routes_panel.results.route",
    "maps_www.routes_panel.form.travel_modes",
    "maps_www.routes_panel.results.send_to_phone",
    "maps_www.collapsed_routes_panel.travel_modes",
]


WEB_MAPS_DEEPUSE_PREFIXES = [
    "maps_www.serp_panel.preview_card.",
    "maps_www.poi_panel.preview_card.",
    "maps_www.bookmarks_panel.preview_card.",
    "maps_www.orgpage.",
    "maps_www.routes_control",
    "maps_www.print",
    "maps_www.print_control",
    "maps_www.tools_list.print",
    "maps_www.share_popup.share_view.code",
    "maps_www.share_popup.share_view.link",
    "maps_www.share_popup.share_view.social",
    "maps_www.actions.build_route",
    "maps_www.collapsed_serp_panel.collapsed_card.actions.",
    "maps_www.collapsed_poi_panel.collapsed_card.actions.",
    "maps_www.collapsed_bookmarks_panel.collapsed_card.actions.",
    "maps_www.mini_card_serp_panel.collapsed_card.",
    "maps_www.mini_card_poi_panel.collapsed_card.",
    "maps_www.mini_card_bookmarks_panel.collapsed_card.",
    "maps_www.panorama_player.placemark_business_preview",
    "maps_www.org_pin_preview",
    "maps_www.org_poi_preview"
]

WEB_MAPS_DEEPUSE_BLACK_LIST = {
    "maps_www.orgpage.content.header.pin",
    "maps_www.orgpage.content.header.all_places",
    "maps_www.orgpage.content.header.search",
    "maps_www.orgpage.content.collapse"
}

MOBILE_MAPS_DEEPUSE_PREFIXES = [
    "place.",
    "add-bookmark.attempt",
    "add-bookmark.submit",
    "add-photo.submit",
    "add-review.attempt",
    "add-review.submit"
]

MOBILE_MAPS_DEEPUSE_BLACK_LIST = {
    "place.add-photo.review-pop",
    "place.add-photo.error",
    "place.ugc-panel.appear",
    "place.ugc-panel.hide"
}

NAVI_DEEPUSE_PREFIXES = [
    "place.",
    "route.make-route"
]

NAVI_DEEPUSE_BLACK_LIST = {
    "place.open-place-view",
    "place.close-place-view"
}


def match_by_prefix_and_black_list(path, prefixes, black_list):
    return any((path.startswith(prefix) for prefix in prefixes)) and path not in black_list


def is_deep_use_mobile_maps_search_lib(click):
    return match_by_prefix_and_black_list(
        click.get("path"),
        MOBILE_MAPS_DEEPUSE_PREFIXES,
        MOBILE_MAPS_DEEPUSE_BLACK_LIST
    )


def is_deep_use(client, click):
    try:
        click = yson.loads((click or {}))
    except:
        pass
    if not bool(click.get("path")):
        return False
    if client == "maps":
        return match_by_prefix_and_black_list(click.get("path"), WEB_MAPS_DEEPUSE_PREFIXES, WEB_MAPS_DEEPUSE_BLACK_LIST) \
            and not any(click.get("path").endswith(end) for end in [".close", ".back", ".collapse", ".header"])
    elif client == "mobile_maps":
        return is_deep_use_mobile_maps_search_lib(click)
    elif client == "navi":
        return match_by_prefix_and_black_list(click.get("path"), NAVI_DEEPUSE_PREFIXES, NAVI_DEEPUSE_BLACK_LIST) or \
            is_deep_use_mobile_maps_search_lib(click)
    else:
        return False


def is_goal_deep_use(client, click):
    try:
        click = yson.loads((click or {}))
    except:
        pass
    actions = parse_click(client, click.get("path"), click.get("vars", {}))
    return bool(set(actions) & set(GOAL_DEEP_USE_ALIASES))


def is_goal_deep_use_by_alias(alias):
    return alias in GOAL_DEEP_USE_ALIASES


def is_routes_deep_use(client, click):
    if not bool(click.get("path")):
        return False
    if client == "maps":
        return match_by_prefix_and_black_list(click.get("path"), WEB_MAPS_ROUTES_DEEPUSE_PREFIXES, {})


def is_search_success(client, click):
    actions = parse_click(client, click.get("path"), click.get("vars", {}))
    return bool(set(actions) & (set(GOAL_DEEP_USE_ALIASES) | {"any_site"}))
