import argparse
import os
import toml

from datetime import datetime, timedelta

from reporter.lib.report import YqlReport
from reporter.lib.job import YqlJob, DATE_FORMAT
from reporter.lib.worker import YqlWorker
from reporter.lib.notificator import Notificator

from library.python import resource


QUERY_FILENAME = 'query.sql'
CONFIG_FILENAME = 'config.toml'

JOBS_TYPES = {
    'reports': YqlReport,
    'jobs': YqlJob
}


def get_items(type_):
    items = {}
    for path in resource.resfs_files():
        if path.startswith('reporter/%s/' % type_):
            filepath, filename = os.path.split(path)
            _, foldername = os.path.split(filepath)

            data = resource.resfs_read(path)

            if foldername not in items:
                items[foldername] = {}

            if filename == QUERY_FILENAME:
                items[foldername]['query'] = data
            elif filename == CONFIG_FILENAME:
                items[foldername]['config'] = toml.loads(data)

    return items


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument('-t', '--yql_token', help='YQL token')
    parser.add_argument('-d', '--date', help='Date')
    parser.add_argument('-s', '--scale', help='Scale')

    parser.add_argument('--name', help='Report name')
    parser.add_argument('--start_date', help='Start date')
    parser.add_argument('--end_date', help='End date')

    parser.add_argument('--auto', help='Do not use this argument', action='store_true')

    args = parser.parse_args()

    if args.auto:
        if not args.date:
            raise ValueError('Use "--date" in auto mode')
    else:
        if not args.name:
            raise ValueError('If it was not started automatically, please specify the report name using "--name"')

    notificator = Notificator(sender='robot-navi-analytics@yandex-team.ru')

    yql_token = args.yql_token or os.environ['YQL_TOKEN']
    worker = YqlWorker(db='hahn', token=yql_token, notificator=notificator)

    for job_type, job_class in JOBS_TYPES.items():
        jobs_items = get_items(job_type)
        jobs = []
        if args.name:
            if args.name in jobs_items:
                jobs = [(args.name, jobs_items[args.name])]
        else:
            jobs = jobs_items.items()

        for name, job in jobs:
            r = job_class(
                title=name,
                config=job['config'],
                query=job['query'],
                yql_worker=worker,
                ignore_latency=not args.auto
            )

            date, start_date, end_date = args.date, args.start_date, args.end_date

            if args.auto and r.latency:
                dt = datetime.strptime(args.date, DATE_FORMAT)
                date = (dt - timedelta(days=r.latency)).strftime(DATE_FORMAT)

            if args.scale is None or r.scale == args.scale:
                r.run(date=date, start=start_date, end=end_date)

    errors = worker.wait_operations()
    if errors:
        raise Exception('Some calculations failed with an error.')

if __name__ == '__main__':
    main()
