from datetime import datetime, timedelta
from dateutil.relativedelta import relativedelta, MO

from reporter.lib.worker import YqlAttachment


_INIT_VARS_QUERY_ = """
PRAGMA yt.Pool = 'stats-navi-analytics';

$__SCALE__ = '{scale}';
$__START_DATE__ = CAST('{start_date}' AS Date);
$__END_DATE__ = CAST('{end_date}' AS Date);
$__LATENCY__ = {latency};"""

DATE_FORMAT = '%Y-%m-%d'


def query_formatter(title, query, fill_char='-', fill_length=80, fill_align='^'):
    return '{header:{fill_char}{fill_align}{fill_length}}\n' \
        '{query}\n' \
        '{bottom:{fill_char}{fill_align}{fill_length}}\n' \
        .format(
            header=' {title} - {comment} '.format(
                title=title, comment='(added automatically)'
            ),
            bottom='',
            query=query.strip(),
            fill_align=fill_align,
            fill_char=fill_char,
            fill_length=fill_length
        )


class JobConfig(object):
    def __init__(self, config, required_keys=()):
        self.__config = config

        for key in required_keys:
            if key not in config:
                raise self.key_exception(key)

    def __contains__(self, key):
        return key in self.__config

    def __getitem__(self, key):
        if key not in self:
            raise self.key_exception(key)

        return self.__config[key]

    def get(self, key, default=None):
        return self[key] if key in self else default

    @staticmethod
    def key_exception(key):
        return KeyError(
            '"{}" parameter must be specified in the report configuration'.format(key)
        )


class YqlJob(object):
    __required_config_params = ('owners', 'scale')

    def __init__(self, title, config, query, yql_worker, ignore_latency):
        self.title = title
        self.query = query
        self.yql_worker = yql_worker
        self.config = JobConfig(config, self.__required_config_params)

        self.owners = config['owners']
        self.scale = config['scale']

        self.latency = 0 if ignore_latency else config.get('latency', 0)

        self.attachments = [
            YqlAttachment(
                path=a['path'],
                name=a['name'],
                type_=a['type']
            ) for a in config['attachments']
        ] if config.get('attachments') else None

    @staticmethod
    def period_by_date(date, scale):
        dt = datetime.strptime(date, DATE_FORMAT)
        if scale == 'd':
            start = end = dt
        elif scale == 'w':
            start = dt + relativedelta(weekday=MO(-1))
            end = start + timedelta(days=6)
        elif scale == 'm':
            start = dt.replace(day=1)
            end = start + relativedelta(months=+1) - timedelta(days=1)
        elif scale == 'y':
            start = dt.replace(day=1, month=1)
            end = start + relativedelta(years=+1) - timedelta(days=1)

        return (start.strftime(DATE_FORMAT), end.strftime(DATE_FORMAT))

    def preprocess_query(self, date=None, start=None, end=None):
        if not (date or start and end):
            raise Exception(
                'Please specify a date '
                'or beginning (start_date) and end (end_date) of the period'
            )

        if date is None:
            start_date, end_date = start, end
        else:
            start_date, end_date = self.period_by_date(date, self.scale)

        global_vars = query_formatter(
            'EXTERNAL VARIABLES',
            _INIT_VARS_QUERY_.format(
                start_date=start_date,
                end_date=end_date,
                scale=self.scale,
                latency=self.latency
            )
        )

        return global_vars + self.query

    def run(self, **kwds):
        q = self.preprocess_query(**kwds)
        self.yql_worker.async_run(
            owners=self.owners,
            query=q,
            title='Reporter: {} [powered by YQL]'.format(self.title),
            attachments=self.attachments
        )
