from reporter.lib.worker import YqlAttachment
from reporter.lib.job import YqlJob, JobConfig, query_formatter

YQL_REPORT_LIB = YqlAttachment(
    path='reporter/yql_lib/report.sql',
    name='report.sql',
    type_='path'
)

_INIT_VARS_QUERY_ = """
PRAGMA AnsiOrderByLimitInUnionAll;
PRAGMA Library("report.sql");
IMPORT report SYMBOLS
    $discardEnsure,
    $region_check,
    $platform_check;

$__LATENCY_INTERVAL__ = DateTime::IntervalFromDays($__LATENCY__);"""

_GET_REPORT_QUERY = """
$__REPORT = (
    SELECT
        *
    FROM
        $__REPORT__($__SCALE__, $__START_DATE__, $__END_DATE__)
);"""

_REPORT_REGION_CHECK_QUERY = """
DISCARD SELECT $discardEnsure(
    $region_check({region_field}),
    "Expected region id"
) FROM $__REPORT;"""

_REPORT_PLATFORM_CHECK_QUERY = """
DISCARD SELECT $discardEnsure(
    $platform_check({platform_field}),
    "Unknown platform type"
) FROM $__REPORT;"""

_INSERT_REPORT_QUERY = """
INSERT INTO `{report_path}` WITH TRUNCATE"""

_HISTORICAL_DATA_QUERY = """
SELECT
    *
FROM `{report_path}`
WHERE
    CAST({date_field} as Date) NOT BETWEEN $__START_DATE__ AND $__END_DATE__
UNION ALL"""


_SELECT_REPORT_QUERY = """
SELECT
    *
FROM
    $__REPORT
WHERE
    CAST({date_field} as Date) BETWEEN $__START_DATE__ AND $__END_DATE__"""

_ORDER_REPORT_QUERY = """
ORDER BY {date_field};"""


class YqlReport(YqlJob):
    __default_tmp_path = '//tmp/navi/reporter/worker'

    __required_config_params = ('owners', 'scale', 'path')

    __date_field = 'fielddate'
    __platform_field = 'platform'
    __region_field = 'region'

    __required_fields = (__date_field, __platform_field, __region_field)

    def __init__(self, title, config, query, yql_worker, ignore_latency):
        super(YqlReport, self).__init__(title, config, query, yql_worker, ignore_latency)

        self.config = JobConfig(config, self.__required_config_params)
        self.path = config['path']

        self.required_fields = [
            f for f in self.__required_fields if config['fields'].get(f, True)
        ]

        # include yql report lib if it was not included erlier
        if YQL_REPORT_LIB not in self.attachments:
            self.attachments.append(
                YQL_REPORT_LIB
            )

    def preprocess_query(self, date=None, start=None, end=None):
        job_preprocessed_query = super(YqlReport, self).preprocess_query(date, start, end)

        report_fields_check_query = '' + \
            _REPORT_REGION_CHECK_QUERY if self.__region_field in self.required_fields else '' + \
            _REPORT_PLATFORM_CHECK_QUERY if self.__platform_field in self.required_fields else ''

        insert_results = _INIT_VARS_QUERY_ + \
            _GET_REPORT_QUERY + \
            report_fields_check_query + \
            _INSERT_REPORT_QUERY + \
            (_HISTORICAL_DATA_QUERY if self.yql_worker.path_exists(self.path) else '') + \
            _SELECT_REPORT_QUERY + \
            _ORDER_REPORT_QUERY

        insert_results = insert_results.format(
            date_field=self.__date_field,
            report_path=self.path,
            region_field=self.__region_field,
            platform_field=self.__platform_field
        )

        return job_preprocessed_query + query_formatter('REPORT EXECUTION', insert_results)
