from yt.wrapper import YtClient

from yql.api.v1.client import YqlClient
from library.python import resource
from yql.config import config


class YqlAttachment(object):
    def __init__(self, path, name, type_):
        self.path = path
        self.name = name
        self.type = type_

    def __eq__(self, other):
        if isinstance(other, YqlAttachment):
            return self.path == other.path and \
                self.name == other.name and \
                self.type == other.type
        return False


class Operation(object):
    def __init__(self, owners, title, request, response):
        self.owners = owners
        self.title = title
        self.request = request
        self.operation_id = response.operation_id
        self.web_url = response.web_url

    def get_results(self):
        return self.request.get_results()


class MrWorker(object):
    def __init__(self):
        super().__init__()

    def run(self):
        pass

    def async_run(self):
        pass


class YqlWorker(MrWorker):
    __syntax_version = 1

    def __init__(self, db=None, token=None, notificator=None):
        self.__notificator = notificator
        self.__yql_client = YqlClient(db=db, token=token)
        self.__yt_client = YtClient(db)

        self.__yt_client.config['api_version'] = 'v3'
        self.__yt_client.config['token'] = config.token

        self.operations = []

    @staticmethod
    def callback(operation):
        print('Operation %s is %s, progress data:\n%s' % (
            operation.operation_id,
            operation.status,
            str(operation.json)
        ))
        if 'status' in operation.json and not operation.in_progress:
            print(operation.json)

    def wait_operations(self):
        errors = 0
        for operation in self.operations:
            result = operation.get_results()

            header = 'Status: {status} | Operation: "{title}"' \
                .format(
                    title=operation.title,
                    status=result.status
                )

            msg = '\n{header} \nID: {op_id} \nURL: {url} \n'.format(
                header=header,
                url=operation.web_url,
                op_id=operation.operation_id
            )
            if not result.is_success:
                if result.errors:
                    msg += 'Returned errors:\n'
                    for error in result.errors:
                        msg += ' - ' + str(error) + '\n'

                self.__notificator.send(
                    operation.owners,
                    '[Reporter] {}'.format(header),
                    msg
                )

            print(msg)
            errors += not result.is_success

        return errors

    def subscribe(self):
        self.__yql_client.subscribe(self.callback)

    def path_exists(self, path):
        return self.__yt_client.exists(path)

    def run(self, owners, query, title=None, attachments=None):
        print('----- run -----')
        request = self.__yql_client.query(
            query=query,
            title=title,
            syntax_version=self.__syntax_version
        )

        if attachments:
            for a in attachments:
                print('attached: %s' % a.path)
                if a.type == 'path':
                    request.attach_file_data(resource.resfs_read(a.path), a.name)
                elif a.type == 'url':
                    request.attach_url(a.path, a.name)

        response = request.run()

        self.operations.append(Operation(owners, title, request, response))

        print('\nLunched operation: {}'.format(response.operation_id))
        print('\nUrl: {}'.format(response.web_url))
        return response

    def async_run(self, *args, **kwds):
        self.run(*args, **kwds)
