PRAGMA yt.InferSchema = '1';
PRAGMA AnsiInForEmptyOrNullableItemsCollections;
PRAGMA yson.AutoConvert;
PRAGMA yson.DisableStrict;

-- CONSTANTS --

$HEALTH_APP_KEY = "3281023";
$LAUNCHER_KEY = "517959";
$API_KEYS = ($HEALTH_APP_KEY, $LAUNCHER_KEY);
$LAUNCHER_MIN_VERSION = '1.8.0';
$TIMEZONE = 'Europe/Moscow';

-- INCLUDES --

PRAGMA Library("metrika_mobile_log.sql");
IMPORT metrika_mobile_log SYMBOLS 
    $metrika_mobile_log;

PRAGMA Library("common.sql");
IMPORT common SYMBOLS 
    $round_period_str;


-- OPERATIONS -- 
use hahn;

$launcher_version_extractor = ($v) -> {
    $extractor = Re2::Capture("\\d+.\\d+.\\d+");
    RETURN $extractor($v)._0;
};

DEFINE SUBQUERY $__REPORT__($scale, $start_date, $end_date) AS
    $logs = (
        SELECT
            api_key,
            app_id,
            $round_period_str(
                AddTimezone(
                    DateTime::FromSeconds(event_timestamp), 
                    $TIMEZONE
                ), 
                $scale
            ) as fielddate,
            String::ToUpper(report_environment['HeadId']) as head_id,
            event_name,
            event_timestamp,
            event_number,
            event_value_raw,
            report_environment,
            $launcher_version_extractor(app_version_raw) as launcher_version
        FROM
            $metrika_mobile_log(
                $API_KEYS, 
                $start_date, $end_date
            )
        WHERE
            event_date BETWEEN $start_date AND $end_date
    );
    
    $drive_traffic_logs = (
        SELECT
            fielddate,
            head_id,
            event_timestamp,
            event_number,
            Yson::ConvertToList(Yson::ConvertToDict(
                CAST(event_value_raw as JSON))['device_values']
            ) as device_values,
            Yson::ConvertToList(Yson::ConvertToDict(
                CAST(event_value_raw as JSON))['uid_values']
            ) as uid_values
        FROM 
            $logs
        WHERE
            event_name == 'network.traffic'
            
            AND head_id || fielddate IN (
                SELECT 
                    head_id_CONCAT_fielddate
                FROM
                    $logs
                WHERE
                    app_id == 'yandex.auto'
                GROUP BY 
                    head_id || fielddate as head_id_CONCAT_fielddate
                HAVING
                    COUNT_IF(launcher_version >= $LAUNCHER_MIN_VERSION) > 0
                    AND COUNT_IF(launcher_version < $LAUNCHER_MIN_VERSION) == 0
            )
    );
    
    $uid_and_device_flatten_logs = (
        SELECT
            fielddate,
            head_id,
            value,
            'uid' as traffic_type,
            event_timestamp,
            event_number,
        FROM 
            $drive_traffic_logs
        FLATTEN LIST BY (
            uid_values as value
        )
        UNION ALL 
        SELECT
            fielddate,
            head_id,
            value,
            'device' as traffic_type,
            event_timestamp,
            event_number,
        FROM 
            $drive_traffic_logs
        FLATTEN LIST BY (
            device_values as value
        )
    );
    
    $traffic_calculation = (
        SELECT
            fielddate,
            head_id,
            event_timestamp,
            event_number,
            
            traffic_type,
            network,
            subscriber_id,
            is_owner,
            package_names,
            package_names_len,
            
            received,
            sent,
            
            IF(LEAD(received) IS NULL OR LEAD(received) < received,
                    received,
                    0
            ) OVER w as guest_received,
            IF(LEAD(sent) IS NULL OR LEAD(sent) < sent,
                sent,
                0
            ) OVER w as guest_sent
        FROM
        (
            SELECT
                fielddate,
                head_id,
                event_timestamp,
                event_number,
                
                traffic_type,
                Yson::ConvertToString(value['network']) as network,
                Yson::ConvertToString(value['subscriber_id']) as subscriber_id,
                Yson::ConvertToBool(value['is_owner']) ?? true as is_owner,
                ListLength(Yson::ConvertToStringList(value['package_names'])) as package_names_len,
                ListConcat(Yson::ConvertToStringList(value['package_names']), ',') as package_names,
                
                Yson::ConvertToInt64(value['received']) as received,
                Yson::ConvertToInt64(value['sent']) as sent
            FROM 
                $uid_and_device_flatten_logs
        )
        WINDOW w As (
            PARTITION BY 
                fielddate,
                head_id,
                traffic_type,
                network,
                subscriber_id,
                package_names,
                is_owner
            ORDER BY 
                event_timestamp,
                event_number
        )
    );
    
    SELECT 
        fielddate,
        head_id,
        
        traffic_type,
        network,
        subscriber_id,
        package_names as uid_package_names,
        package_names_len as uid_package_names_len,
        
        -- owner traffic
        MAX(IF(is_owner, received, 0)) - MIN(IF(is_owner, received, 0))
        -- guest traffic
            + SUM(IF(is_owner, 0, guest_received)) as received,
            
        MAX(IF(is_owner, sent, 0)) - MIN(IF(is_owner, sent, 0))
            + SUM(IF(is_owner, 0, guest_sent)) as sent
    FROM
        $traffic_calculation
    GROUP BY
        fielddate,
        head_id,
        traffic_type,
        network,
        subscriber_id,
        package_names,
        package_names_len
    
END DEFINE;
