"""
Функции для создания диапазонов дат и списка таблиц, соответствующего диапазону дат
"""

import re
import calendar
import datetime

def str2date(str_date):
    return datetime.datetime.strptime(str_date, '%Y%m%d').date()

def previous_day(str_date):
    d = str2date(str_date) - datetime.timedelta(days=1)
    return d.strftime('%Y%m%d')

def daterange(start_date, end_date):
    assert end_date >= start_date
    for n in range(int((end_date - start_date).days) + 1):
        yield start_date + datetime.timedelta(days=n)

def get_tables_by_dates(table_path_pref, start_date, end_date, delim=''):
    if type(start_date)==str:
        start_date = str2date(start_date)
    if type(end_date)==str:
        end_date = str2date(end_date)

    tables = []
    for single_date in daterange(start_date, end_date):
        tables.append('%s%s' % (table_path_pref, single_date.strftime('%Y' + delim + '%m' + delim +'%d')))
    return tables

def get_tables_by_dates_str(table_path_pref, dates, delim=''):
    start_date, end_date = get_dates_begin_end(dates, return_str=False)

    if table_path_pref and table_path_pref[-1] != '/':
        table_path_pref += '/'

    tables = []
    for single_date in daterange(start_date, end_date):
        tables.append('%s%s' % (table_path_pref, single_date.strftime('%Y' + delim + '%m' + delim +'%d')))

    return tables

def iterate_dates_by_dates_str(dates, return_str=True):
    date1, date2 = get_dates_begin_end(dates, return_str=False)
    for d in daterange(date1, date2):
        if return_str:
            yield d.strftime('%Y%m%d')
        else:
            yield d

def get_dates_begin_end(dates, return_str=True):
    """Возвращает дату начала и дату конца периода,
    понимая разные форматы записи дат.

    Пример входа -- как понимается (что возвращает функция)
    20190131 -- один день (20190131, 20190131)
    20190101_20190131 -- диапазон дат (20190101, 20190131)
    20190101_31 -- диапазон дат, год и месяц для второй даты для краткости опущены (20190101, 20190131)
    20190101_0331 -- диапазон дат, повторяющийся год для краткости опущен (20190101, 20190331)
    201901 -- весь месяц целиком, функция сама определяет последний день месяца (20190101, 20190131)
    201901_03 -- несколько месяцев целиком, функция сама определяет последний день месяца (20190101, 20190331)

    Отдельный режим -- выдать несколько последних дней
    14_ -- выдаст диапазон, где последним днём будет сегодняшний, а первый день -- 14 дней до.
    0_ -- выдаст диапазон, где будет только сегодняшний день
    Нельзя отсутпать назад более 999 дней

    По умолчанию возвращает дату в виде строк,
    можно получить datetime.date, если задать параметр return_str=False
    """
    if type(dates) == str:
        if re.match(r'\d{6}$', dates): # month
            date0 = date1 = dates[:6]
            month_end = calendar.monthrange(int(date0[:4]), int(date0[4:6]))[1]
            date0 += '01'
            date1 += '%02d'%month_end
        elif re.match(r'\d{6}_\d{2}$', dates): # month range
            date0 = dates[:6]
            date1 = dates[:4] + dates[7:9]
            assert date0 <= date1
            month_end = calendar.monthrange(int(date1[:4]), int(date1[4:6]))[1]
            date0 += '01'
            date1 += '%02d'%month_end
        elif re.match(r'\d{6}_\d{6}$', dates): # month range over year
            date0 = dates[:6]
            date1 = dates[7:13]
            assert date0 <= date1
            month_end = calendar.monthrange(int(date1[:4]), int(date1[4:6]))[1]
            date0 += '01'
            date1 += '%02d'%month_end
        elif re.match(r'\d{1,3}_$', dates): # days before now till today
            days = int(dates[:-1])
            date1 = datetime.datetime.now()
            date0 = date1 - datetime.timedelta(days=days)
            date0 = date0.strftime('%Y%m%d')
            date1 = date1.strftime('%Y%m%d')
        elif re.match(r'\d{8}_$', dates): # days from given date till today
            date0 = dates[:8]
            date1 = datetime.datetime.now().strftime('%Y%m%d')
        elif re.match(r'\d{6}_$', dates):  # days from given month till today
            date0 = dates[:6] + '01'
            date1 = datetime.datetime.now().strftime('%Y%m%d')
        else:
            date0 = dates[:8]
            if re.match(r'\d{8}$', dates):
                date1 = date0
            elif re.match(r'\d{8}_\d{2}$', dates):
                date1 = dates[:6] + dates[9:11]
            elif re.match(r'\d{8}_\d{4}$', dates):
                date1 = dates[:4] + dates[9:13]
            elif re.match(r'\d{8}_\d{8}$', dates):
                date1 = dates[9:17]
            else:
                raise ValueError('Wrong date range (must be 20170101_20170131), but got ' + dates)
    elif type(dates) == list:
        date0, date1 = dates
    elif isinstance(dates, datetime.datetime):
        date0 = date1 = dates.strftime('%Y%m%d')
    else:
        raise ValueError('dates must be a string or a list, but got %s instead: %s'%(str(type(dates)), str(dates)))
    assert date1 >= date0, (dates, date0, date1)

    if return_str:
        return date0, date1
    return (datetime.datetime.strptime(date0, '%Y%m%d'),
            datetime.datetime.strptime(date1, '%Y%m%d'))
