# -*- coding: UTF-8 -*-
import math
from operator import itemgetter

#################
# distance
##################################

EARTH_RADIUS = 6372795.


def sq(x):
    return x ** 2


def distance_latlon(lat1, lon1, lat2, lon2):
    lat1 = math.radians(lat1)
    lat2 = math.radians(lat2)
    lon1 = math.radians(lon1)
    lon2 = math.radians(lon2)

    return EARTH_RADIUS * 2 * math.asin(math.sqrt(
        (math.sin((lat2 - lat1) / 2)) ** 2 +
        math.cos(lat1) * math.cos(lat2) * ((math.sin((lon2 - lon1) / 2)) ** 2)
    ))


def distance_coords(coord1, coord2):
    lat1 = math.radians(coord1[0])
    lat2 = math.radians(coord2[0])
    lon1 = math.radians(coord1[1])
    lon2 = math.radians(coord2[1])

    return EARTH_RADIUS * 2 * math.asin(math.sqrt(
        (math.sin((lat2 - lat1) / 2)) ** 2 +
        math.cos(lat1) * math.cos(lat2) * ((math.sin((lon2 - lon1) / 2)) ** 2)
    ))


def distance_latlon_near(lat1, lon1, lat2, lon2):
    lat1 = math.radians(lat1)
    lat2 = math.radians(lat2)
    lon1 = math.radians(lon1)
    lon2 = math.radians(lon2)

    return EARTH_RADIUS * math.sqrt(
        (lat2 - lat1) ** 2 +
        ((lon2 - lon1) * math.cos(lat1)) ** 2
    )


def distance_coords_near(coord1, coord2):
    lat1 = math.radians(coord1[0])
    lat2 = math.radians(coord2[0])
    lon1 = math.radians(coord1[1])
    lon2 = math.radians(coord2[1])

    return EARTH_RADIUS * math.sqrt(
        sq(lat2 - lat1) +
        sq((lon2 - lon1) * math.cos(lat1))
    )


def distance_latlon_near_square(lat1, lon1, lat2, lon2):
    lat1 = math.radians(lat1)
    lat2 = math.radians(lat2)
    lon1 = math.radians(lon1)
    lon2 = math.radians(lon2)

    return EARTH_RADIUS * ((lat2 - lat1) ** 2 +
                           ((lon2 - lon1) * math.cos(lat1)) ** 2
                           )


def distance_coords_near_square(coord1, coord2):
    lat1 = math.radians(coord1[0])
    lat2 = math.radians(coord2[0])
    lon1 = math.radians(coord1[1])
    lon2 = math.radians(coord2[1])

    return ((lat2 - lat1) ** 2
            + ((lon2 - lon1) * math.cos(lat1)) ** 2)


def distance_near_calculator(lat1, lon1, meters, none_if_far=False):
    """Считаем расстояние до точки только если оно гарантированно меньше meters,
    иначе не считаем, а возвращаем расстояние meters (или больше)
    Возвращает функцию, которая будет осуществлять эту проверку и возвращать расстояние.
    Смысл в экономии вычислительных ресурсов.
    none_if_far=True приведёт к тому, что для далёких точек будет возращаться None
    """

    lat1 = math.radians(lat1)
    lon1 = math.radians(lon1)

    dLat = meters / EARTH_RADIUS
    dLon = meters / EARTH_RADIUS / math.cos(abs(lat1) + dLat)

    lat_min = lat1 - dLat
    lat_max = lat1 + dLat
    lon_min = lon1 - dLon
    lon_max = lon1 + dLon

    # print("test dLat,dLon = ", meters,
    #      distance_latlon(math.degrees(lat_min), math.degrees(lon1), math.degrees(lat_max), math.degrees(lon1)),
    #      distance_latlon(math.degrees(lat1), math.degrees(lon_min), math.degrees(lat1), math.degrees(lon_max)))

    cos = math.cos(lat1)

    def get_distance(lat, lon):
        # return distance_latlon(math.degrees(lat1), math.degrees(lon1), lat, lon)
        lat = math.radians(lat)
        lon = math.radians(lon)
        # return (lat, lat_max, lat_min, lon, lon_max, lon_min)
        if (lat > lat_max or
                lat < lat_min or
                lon > lon_max or
                lon < lon_min):
            return meters

        return EARTH_RADIUS * math.sqrt(
            sq(lat - lat1) +
            sq((lon - lon1) * cos)
        )

    def get_distance_or_none(lat, lon):
        # return distance_latlon(math.degrees(lat1), math.degrees(lon1), lat, lon)
        lat = math.radians(lat)
        lon = math.radians(lon)
        # return (lat, lat_max, lat_min, lon, lon_max, lon_min)
        if (lat > lat_max or
                lat < lat_min or
                lon > lon_max or
                lon < lon_min):
            return None

        distance = EARTH_RADIUS * math.sqrt(
            sq(lat - lat1) +
            sq((lon - lon1) * cos)
        )
        if distance > meters:
            return None
        return distance

    if none_if_far:
        return get_distance_or_none
    else:
        return get_distance


def is_close_calculator(lat1, lon1, meters):
    """Возвращает функцию, которая проверяет, что расстояние от этой точки до новой точки
    меньше или равно meters.
    Возвращает функцию, которая будет осуществлять эту проверку.
    Смысл в экономии вычислительных ресурсов."""

    assert meters >= 0

    lat1 = math.radians(lat1)
    lon1 = math.radians(lon1)

    dLat = meters / EARTH_RADIUS
    cos = math.cos(lat1)
    dLon = meters / EARTH_RADIUS / cos if cos > 0 else 2 * math.pi

    lat_min = lat1 - dLat
    lat_max = lat1 + dLat
    lon_min = lon1 - dLon
    lon_max = lon1 + dLon

    # print("test dLat,dLon = ", meters,
    #      distance_latlon(math.degrees(lat_min), math.degrees(lon1), math.degrees(lat_max), math.degrees(lon1)),
    #      distance_latlon(math.degrees(lat1), math.degrees(lon_min), math.degrees(lat1), math.degrees(lon_max)))

    # cos = math.cos(lat1)
    border = (meters / EARTH_RADIUS) ** 2

    def is_close(lat, lon):
        # return distance_latlon(math.degrees(lat1), math.degrees(lon1), lat, lon)
        lat = math.radians(lat)
        lon = math.radians(lon)
        # return (lat, lat_max, lat_min, lon, lon_max, lon_min)
        if (lat > lat_max or
                lat < lat_min or
                lon > lon_max or
                lon < lon_min):
            return False
        return (lat - lat1) ** 2 + ((lon - lon1) * cos) ** 2 < border

    return is_close


def is_close_coord_to_segment(coord, coord0, coord1, meters):
    """
    Верно ли, что расстояние от точки coord до отрезка (coord0-coord1) меньше, чем meters
    """

    d0 = distance_coords_near(coord, coord0)
    d1 = distance_coords_near(coord, coord1)
    if d0 <= meters or d1 <= meters:
        return True

    d01 = distance_coords_near(coord0, coord1)

    segment_diagonal = d01 ** 2 + meters ** 2
    if d0 ** 2 > segment_diagonal or d1 ** 2 > segment_diagonal:
        return False

    p = (d0 + d1 + d01) / 2.
    pppp = p * (p - d1) * (p - d0) * (p - d01)
    h = 2 * math.sqrt(pppp) / d01 if pppp > 0 else 0
    if h > meters:
        return False

    return True


#################
# bounds
##################################

# def bounds_for_points_list_of_dict_lat_lon(points):
#     min_lat = min(*point, key=itemgetter(0))
#     min_lon = min(*point, key=itemgetter(1))
#     max_lat = max(*point, key=itemgetter(0))
#     max_lon = max(*point, key=itemgetter(1))
#     return ((min_lat, min_lon), (max_lat, max_lon))
#     # maxLL = points[0][:2]
#     # minLL = points[0][:2]
#     # for i in range(1, len(points)):
#     #     p = points[i]
#     #     if p[0]>maxLL[0]: maxLL[0]=p[0]
#     #     if p[1]>maxLL[1]: maxLL[1]=p[1]
#     #     if p[0]<minLL[0]: minLL[0]=p[0]
#     #     if p[1]<minLL[1]: minLL[1]=p[1]
#     # return [minLL, maxLL]

def bounds_for_points(points):
    min_lat = min(points, key=itemgetter(0))[0]
    min_lon = min(points, key=itemgetter(1))[1]
    max_lat = max(points, key=itemgetter(0))[0]
    max_lon = max(points, key=itemgetter(1))[1]
    return ((min_lat, min_lon), (max_lat, max_lon))
    # maxLL = points[0][:2]
    # minLL = points[0][:2]
    # for i in range(1, len(points)):
    #     p = points[i]
    #     if p[0]>maxLL[0]: maxLL[0]=p[0]
    #     if p[1]>maxLL[1]: maxLL[1]=p[1]
    #     if p[0]<minLL[0]: minLL[0]=p[0]
    #     if p[1]<minLL[1]: minLL[1]=p[1]
    # return [minLL, maxLL]


def bounds_from_string(bounds):
    """Распознаёт bounds в одном из 3-х видов
    1) через пробел или запятую, например, "57.5 35.7 57.8 35.9"
    2) как часть маршрута после rtext, например, "55.701261%2C37.657472~55.764801%2C37.749482" либо отдельно либо вычленяет из целой ссылки
    3) предустановленные города или страны
    """
    bounds_shortnames = {
        "TR": [35.844737, 26.050416, 41.99, 44.64],
        "msk": [55.549741, 37.359468, 55.927053, 37.881318]
    }

    def rearrange(bounds):
        """берём плоский bounds, возращаем [left_bottom, right_top]
        при необходимости переставляем координаты местами"""
        return [[min(bounds[0], bounds[2]), min(bounds[1], bounds[3])],
                [max(bounds[0], bounds[2]), max(bounds[1], bounds[3])]]

    if type(bounds) != str:
        raise Exception('bounds type must be str but "%s" type is %s' % (str(bounds), str(type(bounds))))

    try:  # если bounds есть в словаре сокращений -- возвращаем
        return rearrange(bounds_shortnames[bounds])
    except KeyError:
        pass

    bounds_space_separated = bounds.split(' ')
    if len(bounds_space_separated) == 4:
        return rearrange([float(x) for x in bounds_space_separated])

    bounds_comma_separated = bounds.split(',')
    if len(bounds_comma_separated) == 4:
        return rearrange([float(x) for x in bounds_comma_separated])

    if bounds.startswith("http"):
        cut0 = bounds.index("rtext=") + 6
        cut1 = bounds.find("&", cut0)
        if cut1 < 0:
            bounds = bounds[cut0:]
        else:
            bounds = bounds[cut0:cut1]
    elif bounds.startswith("rtext="):
        bounds = bounds[6:]
    points = bounds.split('~')
    if len(points) >= 2:
        bounds = [[float(x) for x in p.split('%2C')] for p in [points[0], points[-1]]]
        return rearrange(bounds[0] + bounds[1])

    raise Exception("Can't understand bounds string %s" % bounds)


def bounds_sum(b1, b2):
    """
    Строит bbox, который вмещает в себя 2 данных bbox
    """
    return [[min(b1[0][0], b1[1][0], b2[0][0], b2[1][0]), min(b1[0][1], b1[1][1], b2[0][1], b2[1][1])],
            [max(b1[0][0], b1[1][0], b2[0][0], b2[1][0]), max(b1[0][1], b1[1][1], b2[0][1], b2[1][1])]]


def bounds_in_bounds(bInner, bOuter):
    """
    Проверяет что один bbox целиком лежит во втором bbox
    """
    if not (bOuter[0][0] <= bInner[0][0] <= bOuter[1][0]):
        return False
    if not (bOuter[0][0] <= bInner[1][0] <= bOuter[1][0]):
        return False
    if not (bOuter[0][1] <= bInner[0][1] <= bOuter[1][1]):
        return False
    if not (bOuter[0][1] <= bInner[1][1] <= bOuter[1][1]):
        return False
    return True


def bounds_enlarged_meters(bounds_degrees, meters):
    bounds = [[math.radians(x) for x in bounds_degrees[i]] for i in range(2)]

    bounds[0][0] -= meters / EARTH_RADIUS
    bounds[1][0] += meters / EARTH_RADIUS

    bounds[0][1] -= meters / EARTH_RADIUS / math.cos(bounds[0][0])
    bounds[1][1] += meters / EARTH_RADIUS / math.cos(bounds[1][0])

    return [[math.degrees(x) for x in bounds[i]] for i in range(2)]


def bounds_enlarge_by_point(bounds, point):  # меняет исходный bounds
    if point[0] < bounds[0][0]:
        bounds[0][0] = point[0]
    if point[1] < bounds[0][1]:
        bounds[0][1] = point[1]
    if point[0] > bounds[1][0]:
        bounds[1][0] = point[0]
    if point[1] > bounds[1][1]:
        bounds[1][1] = point[1]


def point_in_bounds(p, bounds):
    if ((bounds[0][0] <= p[0] <= bounds[1][0]) and
            (bounds[0][1] <= p[1] <= bounds[1][1])):
        return True
    return False


def bounds_intersection(bbox1, bbox2):
    p = [(bbox1[0][0] + bbox1[1][0]) / 2, (bbox1[0][1] + bbox1[1][1]) / 2]
    d_lat = (bbox1[1][0] - bbox1[0][0]) / 2
    d_lon = (bbox1[1][1] - bbox1[0][1]) / 2
    bbox_big = [[bbox2[0][0] - d_lat,
                 bbox2[0][1] - d_lon],
                [bbox2[1][0] + d_lat,
                 bbox2[1][1] + d_lon]]
    return point_in_bounds(p, bbox_big)


#################
# polygon
##################################
def point_in_polygon(p, poly):
    """Returns True if p is inside poly"""
    n = len(poly)
    inside = False
    x, y = p
    p1x, p1y = poly[0]

    for i in range(n + 1):
        p2x, p2y = poly[i % n]
        if y > min(p1y, p2y):
            if y <= max(p1y, p2y):
                if x <= max(p1x, p2x):
                    if p1y != p2y:
                        xints = (y - p1y) * (p2x - p1x) / (p2y - p1y) + p1x
                    if p1x == p2x or x <= xints:
                        inside = not inside
        p1x, p1y = p2x, p2y

    return inside


def get_polygon_checker(poly):
    bounds = bounds_for_points(poly)

    def polygon_checker(p):
        if not point_in_bounds(p, bounds):
            return False
        return point_in_polygon(p, poly)

    return polygon_checker


def get_multi_polygons_checker(polygons_list):
    bounds = None
    checkers = []
    for poly in polygons_list:
        checkers.append(get_polygon_checker(poly))
        bounds_one = bounds_for_points(poly)
        if bounds:
            bounds = bounds_sum(bounds, bounds_one)
        else:
            bounds = bounds_one
    assert bounds

    def polygons_checker(p):
        if not point_in_bounds(p, bounds):
            return False
        for i, checker in enumerate(checkers):
            if checker(p):
                return True
        return False

    return polygons_checker


def get_polygon_center(poly):
    """Находим центроид полигона (центр по "массе")
    Принцип в том, что считаем центры треугольников, образованные одной фиксированной вершиной
    и далее попарно всеми вершинами.
    Args:
        poly: a list of points, each of which is a list of the form [x, y].
    Returns:
        the centroid of the polygon in the form [x, y].
    Raises:
        ValueError: if poly has less than 3 points or the points are not
                    formatted correctly.
    """
    if len(poly) < 3:
        raise ValueError('polygon has less than 3 points')
    for point in poly:
        if type(point) is not list or 2 != len(point):
            raise ValueError('point is not a list of length 2')

    area_total = 0
    centroid_total = [float(poly[0][0]), float(poly[0][1])]
    for i in range(0, len(poly) - 2):
        # Get points for triangle ABC
        a, b, c = poly[0], poly[i+1], poly[i+2]
        # Calculate the signed area of triangle ABC
        area = ((a[0] * (b[1] - c[1])) +
                (b[0] * (c[1] - a[1])) +
                (c[0] * (a[1] - b[1]))) / 2.0
        # The centroid of the triangle ABC is the average of its three
        # vertices
        centroid = [(a[0] + b[0] + c[0]) / 3.0, (a[1] + b[1] + c[1]) / 3.0]
        # Add triangle ABC's area and centroid to the weighted average
        centroid_total[0] = ((area_total * centroid_total[0]) +
                             (area * centroid[0])) / (area_total + area)
        centroid_total[1] = ((area_total * centroid_total[1]) +
                             (area * centroid[1])) / (area_total + area)
        area_total += area
    return centroid_total


def iterate_poly_segments(poly):
    if len(poly) < 2:
        return
    yield (poly[-1], poly[0])
    for i in range(len(poly) - 1):
        yield (poly[i], poly[i + 1])


def segments_crosses(segment1, segment2):
    def normal_projection(segment, point):
        return ((point[0] - segment[0][0]) * (segment[1][1] - segment[0][1])
                - (point[1] - segment[0][1]) * (segment[1][0] - segment[0][0]))

    if (normal_projection(segment1, segment2[0]) * normal_projection(segment1, segment2[1]) < 0
            and (normal_projection(segment2, segment1[0]) * normal_projection(segment2,
                                                                              segment1[1]) < 0)):
        return True

    return False


def is_polyline_in_polygon(polyline, poly, catch_cross=True):
    """
    Проверяет, пересекается ли данная полилиния с полигоном (находится ли в нём хоть частично,
    даже если она срезает угол полигону, т.е. ни одна точка не в полигоне, то тоже засчитывается)

    :param polyline:
    :param poly:
    :param catch_cross:
    :return:
    """

    for p in polyline:
        if point_in_polygon(p, poly):
            return True

    if catch_cross:
        for i in range(len(polyline) - 1):
            for poly_segment in iterate_poly_segments(poly):
                if segments_crosses(
                        (polyline[i], polyline[i + 1]),
                        poly_segment,
                ):
                    return True

    return False


#################
# other
##################################
def step_meters_by_line(p1, p2, meters):
    """Отступает от точки p1 в сторону p2 на заданное число метров
    (считая землю локально плоской!)
    """

    def between(x1, x2, k):
        return (x2 - x1) * k + x1

    dist = distance_coords(p1, p2)
    k = meters / dist
    return [between(p1[0], p2[0], k),
            between(p1[1], p2[1], k)]


def step_meters_by_heading(point, meters, heading):
    """Отступает от точки point в направлении heading на заданное число метров
    (считая землю локально плоской!)
    heading = 0 на север и далее по часовой стрелке в градусах
    """
    heading = math.radians(heading)
    meters_lat = meters * math.cos(heading)
    meters_lon = meters * math.sin(heading)

    diff_lat = math.degrees(meters_lat / EARTH_RADIUS)
    diff_lon = math.degrees(meters_lon / EARTH_RADIUS / math.cos(math.radians(point[0])))

    return [point[0] + diff_lat, point[1] + diff_lon]


#################
# test
##################################

if __name__ == "__main__":
    print(step_meters_by_heading((0, 0), 1000, 270))
    exit()
    bounds = [[57, 37], [58, 38]]
    big_bounds = bounds_enlarged_meters(bounds, 1000)
    print(big_bounds)
    print(distance_coords(bounds[0], big_bounds[0]))

    exit()
    print("Test coords_lib")
    lat1, lon1, lat2, lon2 = (55.737693, 37.632753, 55.739630, 37.634383)
    dc = distance_near_calculator(lat1, lon1, 1e3)
    print(dc(lat2, lon2))
    print(distance_latlon(lat1, lon1, lat2, lon2))
    exit()

    print(distance_latlon(55.760155, 37.620393, 59.909299, 30.387269))
    print(distance_latlon_near(55.760155, 37.620393, 59.909299, 30.387269))
    dc = distance_near_calculator(55.760155, 37.620393, 1e3)
    print(dc(59.909299, 30.387269))
