# -*- coding: UTF-8 -*-
"""Считаем проезды по сегментам вокруг заданных точек"""
from __future__ import print_function

import re
import os
import sys
import time
import json
import math
import datetime
import argparse
from operator import itemgetter

import yt.wrapper as yt


# from taskParams import *
# sys.path.append(os.path.join(os.path.dirname(os.path.realpath(__file__)), ".."))
# from sys_tools import email_from_config
# sys.path.append(os.path.join(os.path.dirname(os.path.realpath(__file__)), "..", "lib"))
from analytics.geo.tools.geometry.lib.coords_lib import (distance_latlon_near, distance_coords_near,
                        bounds_for_points, bounds_enlarged_meters, point_in_bounds, bounds_intersection
                        )
from analytics.geo.tools.tracks.lib.clids import extend_clids
from analytics.geo.tools.dates.lib.dates_tools import get_tables_by_dates_str

# class CaptureTrackByPoint(self, point, name, direction=None):
#     self.point = point
#     self.clear()

def is_close_point_to_segment(distance, point_coord, seg_coord1, seg_coord2, segment_len=None):
    """можно ли считать данный сегмент близким к данной точке"""
    dist1 = distance_coords_near(seg_coord1, point_coord)
    if dist1 < distance:
        return True
    dist2 = distance_coords_near(seg_coord2, point_coord)
    if dist2 < distance:
        return True

    if segment_len is None:
        segment_len = distance_coords_near(seg_coord1, seg_coord2)

    p = (dist1 + dist2 + segment_len) / 2.
    pppp = p * (p - dist1) * (p - dist2) * (p - segment_len)
    h = 2 * math.sqrt(pppp) / segment_len if pppp > 0 else 0
    edge_diagonal2 = segment_len ** 2 + distance ** 2
    # print("iC2", d0, d1, farFromSection, d01, "h=", h)
    if h <= distance and dist1 ** 2 < edge_diagonal2 and dist2 ** 2 < edge_diagonal2:
        return True
    return False


def capture_track_by_point(track, point,
                           distance, direction=None,
                           return_coords=False):
    """
    track -- [lat, lon, timestamp, length_to_next]
    """
    POINTS_TO_IGNORE_PREVIOUS = 100

    result = []

    def append_result(capture_first_i, capture_last_i):
        # if capture_last_i is None:
        #     return
        point1 = track[capture_first_i]
        point2 = track[capture_last_i]
        travel_time = point2['timestamp'] - point1['timestamp']
        if travel_time == 0:
            return
        if direction is not None:
            dx = distance_latlon_near(point1['lat'], point1['lon'], point1['lat'], point2['lon'])
            dy = distance_latlon_near(point1['lat'], point1['lon'], point2['lat'], point1['lon'])
            # if direction == 'vertical':
            #     if dx > dy * 1.1:
            #         return
            #     direction_fact = 'south' if point1['lat'] < point2['lat'] else 'north'
            # elif direction == 'horizontal':
            #     if dy > dx * 1.1:
            #         return
            #     direction_fact = 'east' if point1['lon'] < point2['lon'] else 'west'
            # else:
            #     raise ValueError(direction)
            direction_fact = direction
            if direction == 'north':
                if dx > dy * 1.1 or point1['lat'] > point2['lat']:
                    return
                # direction_fact = 'south' if point1['lat'] < point2['lat'] else 'north'
            elif direction == 'south':
                if dx > dy * 1.1 or point1['lat'] < point2['lat']:
                    return
            elif direction == 'east':  # восток
                if dy > dx * 1.1 or point1['lon'] > point2['lon']:
                    return
            elif direction == 'west':  # запад
                if dy > dx * 1.1 or point1['lon'] < point2['lon']:
                    return
                # direction_fact = 'east' if point1['lon'] < point2['lon'] else 'west'
            elif direction == 'calc':  # посчитай
                if dy >= dx:
                    if point1['lat'] >= point2['lat']:
                        direction_fact = 'south'
                    else:
                        direction_fact = 'north'
                else:
                    if point1['lon'] >= point2['lon']:
                        direction_fact = 'west'
                    else:
                        direction_fact = 'east'
            else:
                raise ValueError(direction)
        res = {
            'timestamp': point1['timestamp'],
            'travel_time': point2['timestamp'] - point1['timestamp'],
            'distance': distance_latlon_near(point1['lat'], point1['lon'],
                                             point2['lat'], point2['lon']),
        }
        if direction is not None:
            res['direction'] = direction_fact
        if return_coords:
            track_lat_lon_t = list(map(lambda row: [row['lat'], row['lon'], row['timestamp']],
                                       track[capture_first_i:capture_last_i + 1]))
            res['track'] = track_lat_lon_t
        result.append(res)

    capture_first_i = None
    ignore_i = 0
    for track_i, track_point in enumerate(track[:-1]):
        next_point = track[track_i + 1]

        if is_close_point_to_segment(distance, point,
                                     (track_point['lat'], track_point['lon']),
                                     (next_point['lat'], next_point['lon']),
                                     segment_len=track_point['length_to_next']):
            if capture_first_i is None:
                capture_first_i = track_i
        else:
            if capture_first_i is not None:
                if capture_first_i > ignore_i:
                    append_result(capture_first_i, track_i)
                    ignore_i = track_i + POINTS_TO_IGNORE_PREVIOUS
                capture_first_i = None

    return result


class ReduceCaptureTrackByPoints:
    def __init__(self, points_dict, radius, use_clids=None, save_coords=False):
        self.points_dict = points_dict
        self.radius = radius
        self.use_clids = use_clids
        self.save_coords = save_coords
        self.CUT_TRACK_DISTANCE = 300
        # print(list(self.points.values()))
        self.all_points_bounds = bounds_enlarged_meters(
            bounds_for_points(list(self.points_dict.values())),
            self.radius + self.CUT_TRACK_DISTANCE)
        print('all_points_bounds', self.all_points_bounds)

    def capture_track(self, track, add_params):
        if len(track) < 5:
            return
        track_lat_lon = list(zip(map(itemgetter('lat'), track),
                                 map(itemgetter('lon'), track)))
        track_bounds = bounds_for_points(track_lat_lon)

        # print('ct>', track, add_params)
        if not bounds_intersection(self.all_points_bounds, track_bounds):
            return

        for name, point in self.points_dict.items():
            if not point_in_bounds(point, track_bounds):
                continue
            direction = point[2] if len(point) == 3 else None
            for report in capture_track_by_point(track, point, self.radius, direction=direction,
                                                 return_coords=self.save_coords):
                report['clid'] = add_params['clid']
                report['uuid'] = add_params['uuid']
                report['name'] = name
                yield report

    def __call__(self, key, recs):
        CUT_TRACK_SECONDS = 60

        if self.use_clids is not None and key['clid'] not in self.use_clids:
            return

        track = []
        prev_point = None
        for track_point in recs:
            # print('track_point>>>')
            # print(track_point)
            # print('len(track)', len(track))
            if prev_point is not None:
                track_cut = False
                time_from_prev = track_point['timestamp'] - prev_point['timestamp']
                if time_from_prev > CUT_TRACK_SECONDS:
                    track_cut = True
                else:
                    length_from_prev = distance_latlon_near(
                        track_point['lat'], track_point['lon'], prev_point['lat'], prev_point['lon'])
                    if length_from_prev > self.CUT_TRACK_DISTANCE:
                        track_cut = True
                    else:
                        prev_point['length_to_next'] = length_from_prev
                        # print('ltn>', length_from_prev)
                if track_cut:
                    for report in self.capture_track(track, key):
                        yield report
                    track = []
            if len(track) > 1e5:
                track = track[-1000:]
            track.append(track_point)
            prev_point = track_point

        for report in self.capture_track(track, key):
            yield report


def main():
    root_dir = os.path.dirname(os.path.realpath(__file__))

    command_arguments = argparse.ArgumentParser(description='Captures track passed near one of the given points')
    command_arguments.add_argument('--date', "-d", required=True,
                                   help='dates range in the format: 20141231_20150101 or 20140920_1031 or 20140901_03')
    # command_arguments.add_argument('--table', help='table in %s with tracks'%my_folder)
    command_arguments.add_argument('--input', "-i", default="task.csv",
                                   help='file with task points format: name;lat;lon;direction (optional:horizontal/vertical)')
    command_arguments.add_argument('--output', '-o', required=True,
                                   help='yt path to output table or folder (in this case the table name will be date')
    command_arguments.add_argument('--radius', default=50, type=float)
    command_arguments.add_argument('--clid', default="cars",
                                   help='clids of tracks to be used, default "cars", use "all" to get all, or "navi" or "ymaps"')
    command_arguments.add_argument('--save-coords', action='store_true')
    command_arguments.add_argument('--proxy', default='hahn')
    # command_arguments.add_argument('--use_yql', action="store_true")
    # command_arguments.add_argument('--pool', help="yt pool_tree/pool like yamaps/yamaps_dev")

    args = command_arguments.parse_args()

    yt.config.set_proxy(args.proxy)

    use_clids = (None if args.clid == "all" else
        extend_clids([args.clid], missing='fail')
    )

    task_dict = {}
    with open(args.input) as task_file:
        for line in task_file:
            row = line.strip().split(';')
            assert 3 <= len(row) <= 4, row
            name = row[0]
            assert name not in task_dict, name
            lat, lon = map(float, row[1:3])
            if len(row) == 4:
                direction = row[3]
                task_dict[name] = (lat, lon, direction)
            else:
                task_dict[name] = (lat, lon)

    print(len(task_dict), "points in task", args.input)

    if not len(task_dict):
        exit(1)

    # input_table = '//home/maps/jams/production/data/signals/2020-11-11[#0:#10000000]'
    # date_from, date_till = get_dates_begin_end(args.date)
    # assert re.match(r'20\d{6}$', args.date)
    # date_str = args.date[:4] + '-' + args.date[4:6] + '-' + args.date[6:8]
    # date_str =
    input_table = get_tables_by_dates_str('//home/maps/jams/production/data/signals/', args.date, delim='-')
    output_table = yt.create_temp_table()  # '//tmp/mednikov/ctbp_debug'
    print('sort', input_table, output_table)
    yt.run_sort(input_table, output_table, sort_by=['clid', 'uuid', 'timestamp'])
    input_table = output_table
    output_table = args.output + args.date if args.output.endswith('/') else args.output

    # task_dict = {'test': [40.22613, 44.450238]}
    reducer = ReduceCaptureTrackByPoints(task_dict, args.radius,
                                         use_clids=use_clids, save_coords=args.save_coords)
    # with open('points.json') as read_data:
    #     data = [json.loads(line.strip()) for line in read_data]
    # for res in reducer({'clid': 'clid', 'uuid': 'uuid'}, data[:100]):
    #     print(res)
    print(reducer, input_table, output_table)
    yt.run_reduce(reducer, input_table, output_table, reduce_by=['clid', 'uuid'])
    print('Results saved to', output_table)

if __name__ == "__main__":
    main()
