# -*- coding: UTF-8 -*-
"""Считаем проезды по сегментам внутри полигона и время пребывания в полигоне"""

from __future__ import print_function

import os
import re
import sys
import json
import time
import datetime
import collections

from analytics.geo.tools.graph.lib.edges import EdgeData
from analytics.geo.tools.geometry.lib.coords_lib import get_polygon_center, distance_coords_near
# from analytics.geo.tools.graph.lib.filter_edges import get_segments_data
# from analytics.geo.tools.dates.lib.dates_tools import iterate_dates_by_dates_str

if sys.version_info[0]==2:
    from codecs import open

def reduce_calc_time(key, recs):
    first_time = prev_time = None
    marked = False

    for rec in recs:
        the_time = datetime.datetime.strptime(rec['enter_time'][:15], '%Y%m%dT%H%M%S')

        if first_time:
            if (the_time - prev_time).total_seconds() > 10*60:
                yield {'group': key['group'],
                       'date_time': first_time.strftime('%Y%m%dT%H%M%S'),
                       'time_spent': (prev_time - first_time).total_seconds(),
                       'uuid': key['uuid'],
                       'clid': rec['clid'],
                       'marked': marked,
                       }
                first_time = the_time
        else:
            first_time = the_time
        prev_time = datetime.datetime.strptime(rec['leave_time'][:15], '%Y%m%dT%H%M%S')
        if rec.get('marked'):
            marked = True
    if first_time:
        yield {'group': key['group'],
               'date_time': first_time.strftime('%Y%m%dT%H%M%S'),
               'time_spent': (prev_time - first_time).total_seconds(),
               'uuid': key['uuid'],
               'clid': rec['clid'],
               'marked': marked,
               }

def reduce_calc_num(key, recs):
    count = 0
    for rec in recs:
        count += 1

    res = dict(rec)
    for f in ['clid', 'uuid', 'track_start_time', 'enter_time', 'leave_time', 'match_time', 'travel_time']:
        try:
            del res[f]
        except KeyError:
            pass
    res['count'] = count

    yield res

def reduce_calc_unique(key, recs):
    count = 0
    prev_uuid = None
    for rec in recs:
        if rec['uuid'] == prev_uuid:
            continue
        prev_uuid = rec['uuid']
        count += 1

    res = {
        'group': key['group'],
        'count': count,
    }

    yield res

def reduce_get_key(key, recs):
    res = dict(key)
    yield res

def edge_namer(id_name_dict):
    def namer(row):
        if row is None:
            return ['name']
        return [id_name_dict.get(row['persistent_id'], 'unknown')]

    return namer

######################################################################################
def load_points(filename, lonlat=False):
    """
    Загружает именованные точки из json
    в формате {имя: [lat, lon], ...}
    если передать lonlat=True, то {имя: [lon, lat], ...}
    """

    with open(filename, encoding='utf-8') as rf:
        task_dict = json.load(rf)

    if lonlat:
        for k, v in task_dict.items():
            task_dict[k] = [ll[::-1] for ll in v]

    return task_dict


def load_polygons(filename, lonlat=False):
    """
    Загружает полигон из json
    Понимает geojson (расширение файла должно быть .geojson)
    Понимает просто json вида:
        1) {имя: [полигон], ...}
        2) [{"id": имя, "polygon": [{"Lat": широта, "Lon": долгота}, ...]}, ...]
    для json вариантов можно передать lonlat=True, если порядок координат полигона не (lat, lon), а (lon, lat)
    """

    with open(filename, encoding='utf-8') as rf:
        task_dict = json.load(rf)

    if filename.endswith('.geojson'):
        assert not lonlat
        task_dict_fix = {}
        known_ids = set()
        for item in task_dict['features']:
            if item['geometry']['type'] != 'Polygon':
#                 raise Exception('Non Polygon item', item)
                continue
            name = item.get('id', 'none')
            if name in known_ids:
                n = 1
                while '{} #{}'.format(name, n) in known_ids:
                    n += 1
                name = '{} #{}'.format(name, n)
            known_ids.add(name)
            assert len(item['geometry']['coordinates']) == 1
            task_dict_fix[name] = item['geometry']['coordinates'][0]
        task_dict = task_dict_fix
        lonlat = True

    if isinstance(task_dict, list):
        assert not lonlat

        def get_lat_lon(ll):
            if isinstance(ll, dict):
                return [ll['Lat'], ll['Lon']]
            return ll

        task_dict = {item['id']: list(map(get_lat_lon, item['polygon']))
                     for item in task_dict}

    if lonlat:
        for k, v in task_dict.items():
            task_dict[k] = [ll[::-1] for ll in v]

    return task_dict

######################################################################################
def add_value_or_make_list(edit_dict, key, value):
    if key in edit_dict:
        if type(edit_dict[key]) is not list:
            edit_dict[key] = [edit_dict[key]]
        edit_dict[key].append(value)
    else:
        edit_dict[key] = value

def points_dict_to_named_edges(points_dict, radius, enlarge_by_roads=0, enlarge_direction=None):
    edge_data = EdgeData()
    persistent_ids = dict()
    for name, lat_lon in points_dict.items():
        if len(lat_lon)!=2:
            print('Bad lat lon {} for {}'
                  .format(len(lat_lon), name))
        found_persistent_ids = edge_data.get_edges_in_circle(lat_lon, radius)
        if enlarge_by_roads:
            core_ids = edge_data.get_short_ids(found_persistent_ids)
            found_ids = list(short_ids_enlarged_by_roads(edge_data, core_ids, enlarge_by_roads, enlarge_direction))
            found_persistent_ids = edge_data.get_persistent_ids(found_ids)
        for id1 in found_persistent_ids:
            add_value_or_make_list(persistent_ids, id1, name)
    print('persistent_ids found', len(persistent_ids))
    return persistent_ids

def polygons_dict_to_named_edges(polygons_dict, catch='all'):
    edge_data = EdgeData()
    persistent_ids = dict()
    for name, polygon in polygons_dict.items():
        if len(polygon)<=2:
            print('Bad polygon (only {} points): {}'
                  .format(len(polygon), name))
        for id1 in edge_data.get_edges_in_polygon(polygon, catch=catch):
            persistent_ids[id1] = name
    print('persistent_ids found', len(persistent_ids))
    return persistent_ids

def polygons_dict_enlarged_to_named_edges(polygons_dict, enlarge_meters):
    """
    Для каждого полигона находим центр, находим круг, в который помещается полигон,
        расширяем его на enlarge_meters (может быть и 0),
        и выгружаем persistent_id для этого круга

    Возвращаем словарь с ключом persistent_id,
        а в качестве значений может быть либо одно имя (ключ из polygons_dict),
        либо список имён (если расширенные круги вокруг полигонов расширяются
        и ребро попадает в несколько)
    """
    edge_data = EdgeData()
    persistent_ids = dict()
    for name, polygon in polygons_dict.items():
        if len(polygon)<=2:
            print('Bad polygon (only {} points): {}'
                  .format(len(polygon), name))
        center = get_polygon_center(polygon)
        radius = max(distance_coords_near(center, p)
                     for p in polygon)
        radius += enlarge_meters
        for id1 in edge_data.get_edges_in_circle(center, radius):
            add_value_or_make_list(persistent_ids, id1, name)
#             if id1 in persistent_ids:
#                 if type(persistent_ids[id1]) is not list:
#                     persistent_ids[id1] = [persistent_ids[id1]]
#                 persistent_ids[id1].append(name)
#             else:
#                 persistent_ids[id1] = name
    print('persistent_ids found', len(persistent_ids))
    return persistent_ids

def short_ids_enlarged_by_roads(edge_data, core_ids, enlarge_meters, direction):
    if direction == 'back':
        get_connected_edges = edge_data.get_in_edges_short
    elif direction == 'forward':
        get_connected_edges = edge_data.get_out_edges_short
    else:
        raise ValueError(direction)

    ids_path_dict = {sid: 0 for sid in core_ids}
    found_ids = set(core_ids)
    front_ids = collections.deque(core_ids)
    while front_ids:
        sid = front_ids.popleft()
        path = ids_path_dict[sid]
        for connected_sid in get_connected_edges(sid):
            new_path = path + edge_data.get_edge_length(short_id=connected_sid)
            if connected_sid in ids_path_dict:
                old_path = ids_path_dict[connected_sid]
                if old_path <= new_path:
                    continue
            ids_path_dict[connected_sid] = new_path
            found_ids.add(connected_sid)
            if new_path < enlarge_meters and connected_sid not in front_ids:
                front_ids.append(connected_sid)
    return found_ids

def polygons_dict_enlarged_roads_to_named_edges(polygons_dict, enlarge_meters, direction, edge_data=None):
    """
    От данного полигона ищем дороги, въезжающие в него (но не более enalarge_meters)
    На входе ожидаем dict: имя полигона -> спискок координат для каждого полигона.
    На выходе dict: persistent_id -> имя полигона, а если одно ребро входит в несколько, то list имён полигонов
    """

    if edge_data is None:
        edge_data = EdgeData()
    persistent_ids = dict()
#     with open('debug.json', 'w') as debug_file:
    for name, polygon in polygons_dict.items():
        if len(polygon)<=2:
            print('Bad polygon (only {} points): {}'
                  .format(len(polygon), name))

        core_ids = edge_data.get_short_ids(edge_data.get_edges_in_polygon(polygon))
        found_ids = list(short_ids_enlarged_by_roads(edge_data, core_ids, enlarge_meters, direction))
        found_persistent_ids = edge_data.get_persistent_ids(found_ids)
#         print('-'*30)
#         print(found_ids)
#         print(found_persistent_ids)

#         json.dump({'name': name,
#                    'polygon': polygon,
#                    'edges': {found_ids[i]: edge_data.get_edge_geometry(pid)
#                              for i, pid in enumerate(found_persistent_ids)},
#                    },
#                   debug_file,
#                   )
#         debug_file.write('\n')

        for id1 in set(found_persistent_ids):
            if id1 in persistent_ids:
                if type(persistent_ids[id1]) is not list:
                    persistent_ids[id1] = [persistent_ids[id1]]
                persistent_ids[id1].append(name)
            else:
                persistent_ids[id1] = name
    print('persistent_ids found', len(persistent_ids))
    return persistent_ids
