from nile.api.v1 import (
        Record,
        files,
        cli,
        with_hints,
)

import re


def get_ui(r):
    ui = ''
    if r.IsA('TDesktopUIProperties'):
        ui = 'desktop'
    elif r.IsA('TTouchUIProperties'):
        ui = 'touch'
    elif r.IsA('TMobileUIProperties'):
        ui = 'mobile'
    elif r.IsA('TMobileAppUIProperties'):
        ui = 'mobileapp'
    elif r.IsA('TPadUIProperties'):
        ui = 'pad'
    elif r.IsA('TSiteSearchUIProperties'):
        ui = 'sitesearch'
    return ui


def is_images_greenurl(path_list, varlist):
    is_gu = False
    clickSource=''
    dataSource=''
    for var in varlist:
        if var[0] == "-action-type" and var[1] == "greenurl":
            is_gu = True
        if var[0] == "-click-source":
            clickSource = '/'+var[1]
        if var[0] == "-data-source":
            dataSource = '/'+var[1]

    if is_gu:
        path_list[0] = path_list[0] + dataSource + clickSource
        return True

    if path_list[0] is None:
        return False
    PATTERN_GREENURL = re.compile(
            r"^/image/.*/(site|url|title|link|button|snippet|collections|commercial/(incut/)?(similar|description|related|text|contacts|sitelink|((incut|behavioral)/)?thumb)|(market_offers|direct|polaroid/market)/click|(duplicates/(rating|price|model-rating|button|title|url)/(market|schemaorg)))")
    if PATTERN_GREENURL.search(path_list[0]):
        return True
    else:
        return False


def parse_greenurls(actions):
    for key, records in actions:
        try:
            import libra
            uid = key.key
            sessions = libra.ParseSession(records, 'blockstat.dict')
        except (NameError, AttributeError, TypeError):
            raise
        except Exception:
            continue

        for request in sessions:
            cm2 = 0
            vcomm = 0
            if not request.IsA('TImagesRequestProperties'):
                continue
            reqid = request.ReqID
            query = request.Query

            if hasattr(request, 'RelevValues'):
                cm2 = float(request.RelevValues.get('cm2', 0))
                vcomm = float(request.RelevValues.get('vcomm', 0))
            for event in request.GetOwnEvents():
                timestamp = str(event.Timestamp)
                tempList = [event.ConvertedPath]
                if event.IsA('TClick'):
                    is_greenurl = is_images_greenurl(tempList, event.GetVars())
                else:
                    is_greenurl = is_images_greenurl(tempList, event.Vars)
                if is_greenurl:
                    yield Record(uid=uid,
                                 reqid=reqid,
                                 timestamp=timestamp,
                                 cm2=cm2,
                                 vcomm=vcomm,
                                 path=tempList[0],
                                 ui=get_ui(request),
                                 query=query,
                                 userRegion=request.UserRegion)


@cli.statinfra_job(options=[cli.Option('date', default='?')])
def make_job(job, nirvana, options):
    job = job.env(
            yt_spec_defaults=dict(
                pool_trees=["physical"],
                tentative_pool_trees=["cloud"]
                ),
            templates=dict(
                tmp_files='//home/images/dev/anastasiiait/jobs_tmp/abt_purchase_' + options.date
            )
    )

    log = job.table('//user_sessions/pub/images/daily/@date/clean')
    log.groupby('key').sort('subkey') \
        .reduce(
                parse_greenurls,
                memory_limit=3*1024,
                files=[
                    files.RemoteFile('//statbox/resources/libra.so'),
                    files.RemoteFile('//statbox/statbox-dict-last/blockstat.dict')
                ]
                )\
        .put(nirvana.output_tables[0])
    return job


if __name__ == '__main__':
    cli.run()
