from nile.api.v1 import (
        Record,
        files,
        cli,
        with_hints
)

import re


def get_ui(r):
    ui = ''
    if r.IsA('TDesktopUIProperties'):
        ui = 'desktop'
    elif r.IsA('TTouchUIProperties'):
        ui = 'touch'
    elif r.IsA('TMobileUIProperties'):
        ui = 'mobile'
    elif r.IsA('TMobileAppUIProperties'):
        ui = 'mobileapp'
    elif r.IsA('TPadUIProperties'):
        ui = 'pad'
    elif r.IsA('TSiteSearchUIProperties'):
        ui = 'sitesearch'
    return ui


def print_baobab_path(block):
    path = [block.name]
    parent = block.parent
    while parent is not None:
        path = [parent.name] + path
        parent = parent.parent
        return ','.join(path)


def is_images_greenurl(path):
    if 'popup' in path:
        return False
    if 'navig' in path:
        return False
    PATTERN_GREENURL = re.compile(r"^/image/.*/(site|url|title|link|button|snippet|collections|commercial/(incut/)?(similar|description|related|text|contacts|sitelink|((incut|behavioral)/)?thumb)|(market_offers|direct)/click|(duplicates/(rating|price|model-rating|button|title|url)/(market|schemaorg)))")
    if PATTERN_GREENURL.search(path):
        return True
    elif path in ['/image/touch/preview/duplicates/title/schemaorg', '/image/touch/preview/duplicates/url/schemaorg', '/image/touch/preview/duplicates/price/schemaorg']:
        return True
    else:
        return False


class RequestFilter(object):

    def __call__(self, actions):
        import libra
        import baobab
        import tamus

        class RuleChecker(object):
            def __init__(self):
                self.rules = {
                    'wizard_images': '$result [@type = "wizard" and @wizard_name = "images"]',
                    'images_viewer': 'images-viewer | gallery/scroller | images/scroller | #wizard_images//images/image | #wizard_images//images/fold/image',
                    'images_viewer_modal': 'MMViewerModal',
                    'images_viewer_direct_snippet': '#images_viewer_modal//pane/MMDirectSnippet | #images_viewer_modal/MMDirectSnippet | #images_viewer_modal/DirectStripe | #images_viewer//direct-snippet | #images_viewer//behavioral-direct',
                    'images_viewer_market_snippet': '#images_viewer_modal//pane/MarketSnippet | #images_viewer//market-snippet',
                    'images_viewer_schemaorg_snippet': '#images_viewer//schemaorg-snippet',
                    'images_viewer_similar_offers_snippet': '#images_viewer_modal//pane/similar-offers | #images_viewer//offers-incut',
                    'images_viewer_snippet': '#images_viewer_direct_snippet | #images_viewer_market_snippet | #images_viewer_similar_offers_snippet | images_viewer_schemaorg_snippet'
                }
                self.all_marks = None

                self.IMG_WIZ_TOUCH_BAOBAB_GREENURL = ["link", "other-sites", "collection"]
                self.IMG_WIZ_DESKTOP_BAOBAB_GREENURL = ["urlnav", "title", "collection-go", "path", "link"]
                self.IMG_WIZ_DESKTOP_BAOBAB_SNIPPET_GREENURL = ["image", "button", "urlnav", "title", "sitelink", "contact", "path", "MMDirectSnippet-StripeMain",
                                                                "MMDirectSnippet-StripeFavicon", "MMDirectSnippet-StripeButton", "MMDirectSnippet-StripeLabel", "DirectStripe"]
                self.IMG_WIZ_TOUCH_BAOBAB_SNIPPET_GREENURL = ["thumb", "title", "url", "price", "market-snippet", "direct-snippet", "behavioral-direct", "schemaorg-snippet"]

            def init_marks(self, joiners):
                self.all_marks = tamus.check_rules_multiple_joiners(self.rules, joiners)

            def is_good_web_images_block(self, block):
                return self.all_marks.has_marker_in_block_with_ancestors(block, 'wizard_images') and self.all_marks.has_marker_in_block_with_ancestors(block, 'images_viewer')

            def is_images_greenurl_on_web(self, block, joiners):
                if block is None:
                    return None

                self.init_marks(joiners)
                if not self.is_good_web_images_block(block):
                    return None

                isInMarket = self.all_marks.has_marker_in_block_with_ancestors(block, 'images_viewer_market_snippet')
                isInSchema = self.all_marks.has_marker_in_block_with_ancestors(block, 'images_viewer_schemaorg_snippet')
                isInDirect = self.all_marks.has_marker_in_block_with_ancestors(block, 'images_viewer_direct_snippet')
                isInSimOf = self.all_marks.has_marker_in_block_with_ancestors(block, 'images_viewer_similar_offers_snippet')
                isInViewerModal = self.all_marks.has_marker_in_block_with_ancestors(block, 'images_viewer_modal')
                isInWizard = self.all_marks.has_marker_in_block_with_ancestors(block, 'wizard_images')

                blockName = block.name

                if isInMarket and (blockName in self.IMG_WIZ_DESKTOP_BAOBAB_SNIPPET_GREENURL or blockName in self.IMG_WIZ_TOUCH_BAOBAB_SNIPPET_GREENURL):
                    return [blockName, 'market']
                if isInSchema and (blockName in self.IMG_WIZ_DESKTOP_BAOBAB_SNIPPET_GREENURL or blockName in self.IMG_WIZ_TOUCH_BAOBAB_SNIPPET_GREENURL):
                    return [blockName, 'schemaorg']
                if isInDirect and (blockName in self.IMG_WIZ_DESKTOP_BAOBAB_SNIPPET_GREENURL or blockName in self.IMG_WIZ_TOUCH_BAOBAB_SNIPPET_GREENURL):
                    return [blockName, 'direct']
                if isInSimOf and (blockName in self.IMG_WIZ_DESKTOP_BAOBAB_SNIPPET_GREENURL or blockName in self.IMG_WIZ_TOUCH_BAOBAB_SNIPPET_GREENURL):
                    return [blockName, 'simillar_offers']
                if (blockName in self.IMG_WIZ_TOUCH_BAOBAB_GREENURL or blockName in self.IMG_WIZ_DESKTOP_BAOBAB_GREENURL) and (isInViewerModal or isInWizard):
                    return [blockName, 'organic']

                return None
        Checker = RuleChecker()
        for key, records in actions:
            uid = key.key
            try:
                session = libra.ParseSession(records, 'blockstat.dict', None)
            except (NameError, AttributeError, TypeError):
                raise
            except Exception:
                continue
            for r in session:
                if not r.IsA('TWebRequestProperties'):
                    continue

                ui = get_ui(r)
                query = r.Query
                joiners = r.BaobabAllTrees()

                if not joiners:
                    continue

                for click in r.GetClicks():
                    if click.IsA("TBaobabClickProperties"):
                        block = click.BaobabBlock(joiners)
                        guBlock = Checker.is_images_greenurl_on_web(block, joiners)

                        if guBlock is not None:
                            yield Record(
                                    uid=uid,
                                    reqid=r.ReqID,
                                    ui=ui,
                                    vcomm=float(r.RelevValues.get('vcomm', '0')),
                                    cm2=float(r.RelevValues.get('cm2', '0')),
                                    path=guBlock[1] + ': ' + guBlock[0],
                                    full_path=print_baobab_path(block),
                                    ts=click.Timestamp,
                                    query=query,
                                    userRegion=r.UserRegion
                            )


@cli.statinfra_job(options=[cli.Option('date', default='?')])
def make_job(job, nirvana, options):
    job_root = nirvana.output_tables[0]

    hints = with_hints(output_schema=dict(
                               uid=str,
                               reqid=str,
                               ui=str,
                               vcomm=float,
                               cm2=float,
                               path=str,
                               full_path=str,
                               ts=int,
                               query=str,
                               userRegion=int))
    reducer = hints(RequestFilter())
    job = job.env(
            yt_spec_defaults=dict(
                pool_trees=["physical"],
                tentative_pool_trees=["cloud"]
                ),
            templates=dict(
                job_root=job_root
            )
    )

    log = job.table('//user_sessions/pub/search/daily/@dates/clean')
    log.groupby('key').sort('subkey') \
       .reduce(
               reducer,
               memory_limit=3*1024,
               files=[
                   files.RemoteFile('//statbox/resources/libra.so'),
                   files.RemoteFile('//statbox/statbox-dict-last/blockstat.dict')
                   ]
               )\
        .put(nirvana.output_tables[0])
    return job


if __name__ == '__main__':
    cli.run()
