import random
from itertools import groupby

options_keys = ['image_crc', 'n_wins', 'serp_name', 'query_date']
common_keys = ['device', 'query', 'region_id', 'platform', 'queryfresh']
honeypot_keys = ['image_mds', 'image_url', 'page_screenshot_mds', 'page_url', 'snippet', 'title']


def make_honeypot_from_pair(pair, swap):
    if swap:
        pair.reverse()

    result = {}
    result['inputValues'] = {}
    result['outputValues'] = {}
    result['left_options'] = {}
    result['right_options'] = {}

    result['left_options']['is_support_point'] = not swap
    result['right_options']['is_support_point'] = swap

    if pair[0]['n_wins'] > pair[1]['n_wins']:
        result['outputValues']['label'] = 'left'
    else:
        result['outputValues']['label'] = 'right'

    for key in set(pair[0].keys()) & set(pair[1].keys()):
        if key in common_keys:
            result['inputValues'][key] = pair[0][key]
        elif key in options_keys:
            result['left_options'][key] = pair[0][key]
            result['right_options'][key] = pair[1][key]
        elif key in honeypot_keys:
            result['inputValues']['left_' + key] = pair[0][key]
            result['inputValues']['right_' + key] = pair[1][key]

    return result


def make_honeypot_from_pair_vq(pair, swap):
    if swap:
        pair.reverse()

    result = {}
    result['inputValues'] = {}
    result['outputValues'] = {}
    result['left_options'] = {}
    result['right_options'] = {}

    result['left_options']['is_support_point'] = not swap
    result['right_options']['is_support_point'] = swap

    if (swap and pair[1]['options']['support_point']['rank'] == 1) or \
       (not swap and pair[0]['options']['support_point']['rank'] == 3):
        result['outputValues']['label'] = 'right'
    else:
        result['outputValues']['label'] = 'left'

    for key in set(pair[0].keys()) & set(pair[1].keys()):
        if key in common_keys:
            result['inputValues'][key] = pair[0][key]
        elif key in options_keys:
            result['left_options'][key] = pair[0][key]
            result['right_options'][key] = pair[1][key]
        elif key in honeypot_keys:
            result['inputValues']['left_' + key] = pair[0][key]
            result['inputValues']['right_' + key] = pair[1][key]

    result['left_options']['relevance'] = pair[0].get('relevance', '')
    result['right_options']['relevance'] = pair[1].get('relevance', '')

    return result


def make_honeypot_random(pair, swap):
    if swap:
        pair.reverse()

    result = {}
    result['left_options'] = {}
    result['right_options'] = {}

    result['left_options']['is_support_point'] = not swap
    result['right_options']['is_support_point'] = swap

    for key in set(pair[0].keys()) & set(pair[1].keys()):
        if key in common_keys:
            result[key] = pair[0][key]
        elif key in options_keys:
            result['left_options'][key] = pair[0][key]
            result['right_options'][key] = pair[1][key]
        elif key in honeypot_keys:
            result['left_' + key] = pair[0][key]
            result['right_' + key] = pair[1][key]

    result['left_options']['relevance'] = pair[0].get('relevance', '')
    result['right_options']['relevance'] = pair[1].get('relevance', '')

    return {'inputValues': result}


def generate_honeypots(data):
    out_data = []
    key_func = lambda x: (x['query'], x['region_id'], x['platform'])

    data.sort(key=key_func)

    for key, group in groupby(data, key=key_func):
        query, region_id, platform = key
        group_lst = sorted(list(group), key=lambda x: x['n_wins'], reverse=True)

        if len(group_lst) < 3:
            continue
        else:
            max_wins_honeypot = [group_lst[0], group_lst[-2]]
            min_wins_honeypot = [group_lst[-1], group_lst[1]]

        swap = random.choice([True, False])

        out_data.append(make_honeypot_from_pair(max_wins_honeypot, swap))
        out_data.append(make_honeypot_from_pair(min_wins_honeypot, swap))

    return out_data


def generate_honeypots_from_vq(data):
    out_data = []

    for elem in data:
        documents = elem['documents']['documents']
        support_points = elem['support_points']['documents']

        documents.sort(key=lambda x: x['visual_quality'])

        if documents[-1]['visual_quality'] - documents[0]['visual_quality'] < 0.5:
            continue

        support_points.sort(key=lambda x: x['options']['support_point']['rank'])

        max_wins_honeypot = [support_points[0], documents[0]]
        min_wins_honeypot = [support_points[-1], documents[-1]]

        swap = random.choice([True, False])

        if max_wins_honeypot[1]['visual_quality'] < 0.3:
            out_data.append(make_honeypot_from_pair_vq(max_wins_honeypot, swap))
        if min_wins_honeypot[1]['visual_quality'] > 0.7:
            out_data.append(make_honeypot_from_pair_vq(min_wins_honeypot, swap))

    return out_data


def generate_honeypots_random(data, honeypots_count=2):
    out_data = []

    for elem in data:
        honeypots = []

        documents = elem['documents']['documents']
        support_points = elem['support_points']['documents']

        for i in range(honeypots_count):
            honeypot = [random.choice(support_points), random.choice(documents)]
            if honeypot not in honeypots:
                honeypots.append(honeypot)

        for honeypot in honeypots:
          swap = random.choice([True, False])
          out_data.append(make_honeypot_random(honeypot, swap))

    return out_data
