from collections import defaultdict
import sys
import logging

logging.basicConfig(format='%(filename)s[LINE:%(lineno)d]# %(levelname)-8s [%(asctime)s]  %(message)s', level=logging.DEBUG)

COEFF_WIN = 5000
COEFF_BUCHHOLZ = 200
COEFF_INIT = 50

def calc_dbd_scores_array(serp, was_game, win_scores):
    dbd_scores = [0] * len(serp['components'])

    for doc_i in range(len(serp['components'])):
        doc_init_score = serp['init_scores'][doc_i]
        doc_win_score = win_scores[doc_i]

        doc_buchholz_score = 0
        for opponent_i in was_game[doc_i]:
            doc_buchholz_score += win_scores[opponent_i]

        dbd_scores[doc_i] = COEFF_WIN * doc_win_score + COEFF_BUCHHOLZ * doc_buchholz_score + COEFF_INIT * doc_init_score

    return dbd_scores


def generate_pairs_for_next_round(data):
    logging.info("Start")
    out_data = []

    for serp_i, serp in enumerate(data):
        logging.info("Serp {} / {}".format(serp_i + 1, len(data)))

        was_game = defaultdict(set)
        win_scores = [0] * len(serp['components'])
        dbd_scores = serp['dbd_scores']
        taken_for_pair = set()

        logging.info("Loop through {} rounds, set wins".format(len(serp['rounds'])))

        for cur_round in serp['rounds']:
            for game in cur_round:
                win_scores[game['i']] += game['score']
                win_scores[game['j']] += game['n_games'] - game['score']

                was_game[game['i']].add(game['j'])
                was_game[game['j']].add(game['i'])
        logging.info("Win scores: {}".format(win_scores))
        logging.info("Was game: {}".format(was_game))

        docs_ordering = sorted(enumerate(dbd_scores), key=lambda x: x[1], reverse=True) # descending by dbd_score [(doc_i, dbd_score)]
        logging.info("Components ordering: {}".format(docs_ordering))

        pairs_for_next_round = []

        logging.info("Pairs selection for the next round")
        for pos, (doc_i, doc_dbd_score) in enumerate(docs_ordering):
            logging.info("First in pair: {}, score {}".format(doc_i, doc_dbd_score))
            if doc_i in taken_for_pair:
                logging.info("Document is already taken, continue")
                continue
            for cand_i, cand_dbd_score in docs_ordering[pos + 1:]:
                if cand_i in taken_for_pair:
                    logging.info("Candidate {} is already taken, continue".format(cand_i))
                    continue
                if doc_i in was_game[cand_i] or cand_i in was_game[doc_i]:
                    logging.info("Candidate {} was played with earlier, continue".format(cand_i))
                    continue
                logging.info("Second in pair: {}, score {}".format(cand_i, cand_dbd_score))
                pairs_for_next_round.append({'i': serp['components'][doc_i],
                                            'j': serp['components'][cand_i],
                                            'score_i': doc_dbd_score,
                                            'score_j': cand_dbd_score})
                taken_for_pair.add(doc_i)
                taken_for_pair.add(cand_i)
                break

        logging.info("Check for documents with no pairs")
        unmatched_cnt = 0
        for doc_i in range(len(docs_ordering)):
            if doc_i in taken_for_pair:
                continue
            unmatched_cnt += 1
            logging.error("Document {} has no matched pair".format(doc_i))
        logging.info("Total unmatched pairs: {}".format(unmatched_cnt))

        if len(pairs_for_next_round) > 0:
            out_data.append({'query': serp['query'], 'pairs': pairs_for_next_round})
        else:
            logging.error("No pairs selected for query")

    logging.info("Success")
    return out_data


def update_tournament_with_round_games(tournament_data, markup_data):
    logging.info("Start")

    map_document_to_tournament_index = {}

    logging.info("Start document to index map construction")
    for serp_i, serp in enumerate(tournament_data):
        query = serp['query']
        for comp_i, comp in enumerate(serp['components']):
            key = (query['query'], query['region_id'], query['platform'], comp['image_url'], comp['page_url'])
            map_document_to_tournament_index[key] = (serp_i, comp_i)

    logging.info("Append empty game round")
    for serp in tournament_data:
        serp['rounds'].append([])

    logging.info("Fill in round games")
    for game in markup_data:
        left_key = (game['query'], game['region_id'], game['platform'], game['left_image_url'], game['left_page_url'])
        right_key = (game['query'], game['region_id'], game['platform'], game['right_image_url'], game['right_page_url'])

        serp_i, left_comp_i = map_document_to_tournament_index[left_key]
        _, right_comp_i = map_document_to_tournament_index[right_key]

        winning_label = game['label']
        winning_prob = game['probability']
        n_games = game['overlap'] * 2

        if winning_label == 'left':
            score = int(float(winning_prob) * n_games + 0.01)
        else:
            score = n_games - int(float(winning_prob) * n_games + 0.01)

        tournament_data[serp_i]['rounds'][-1].append({
                                                    'i': left_comp_i,
                                                    'j': right_comp_i,
                                                    'score': score,
                                                    'n_games': n_games
                                                    })

    logging.info("Calc dbd score")
    for serp_i, serp in enumerate(tournament_data):
        logging.info("Serp {} / {}".format(serp_i + 1, len(tournament_data)))

        was_game = defaultdict(set)
        win_scores = [0] * len(serp['components'])
        dbd_scores = [0] * len(serp['components'])
        taken_for_pair = set()

        logging.info("Loop through {} rounds, set wins".format(len(serp['rounds'])))

        for cur_round in serp['rounds']:
            for game in cur_round:
                win_scores[game['i']] += game['score']
                win_scores[game['j']] += game['n_games'] - game['score']

                was_game[game['i']].add(game['j'])
                was_game[game['j']].add(game['i'])

        logging.info("Loop through {} components, calc dbd score".format(len(serp['components'])))

        serp['dbd_scores'] = calc_dbd_scores_array(serp, was_game, win_scores)

    logging.info("Success")
    return tournament_data
