#!/usr/bin/env python
# -*- coding: utf-8 -*-

import json
from yt.wrapper import YtClient
from qb2.api.v1.typing import *

from nile.api.v1 import (
    cli,
    files,
    Record,
    with_hints,
    filters as nf,
    statface as ns,
    extractors as ne,
    aggregators as na,
)


def get_pages():
    res = []
    client = YtClient(proxy="hahn")
    export_page = json.loads(client.read_file("//statbox/statbox-dict-last/export_page").read())
    for key, value in export_page.iteritems():
        name = value.get("Name")
        if name is not None:
            if "images.yandex" in name or "gorsel.yandex" in name:
                res.append(key)
    return res


def generate_all_keys(keys, index=0):
    if len(keys) > index:
        for key in generate_all_keys(keys, index + 1):
            cur = keys[index]
            if not isinstance(cur, list):
                cur = [cur]
            for k in cur:
                yield [k] + key
            yield ["_total_"] + key
    else:
        yield []


class ParseCheventImg(object):
    def __init__(self, pages):
        self.pageId_list_img = pages

    def __call__(self, recs):
        for rec in recs:
            pageid = rec.pageid

            if pageid not in self.pageId_list_img:
                continue

            if rec.countertype == "2":
                click, show = 1, 0
            else:
                click, show = 0, 1

            cost = click * float(rec.eventcost) * 30 / 1000000

            devicetype = int(rec.devicetype)
            if devicetype < 4:
                devicetypestr = "touch"
            elif devicetype == 4:
                devicetypestr = "pad"
            else:
                devicetypestr = "desktop"

            for pageid, devicetypestr in generate_all_keys([pageid, devicetypestr]):
                yield Record(
                    show=show,
                    cost=cost,
                    click=click,
                    page_id=pageid,
                    device=devicetypestr,
                )


@cli.statinfra_job
def make_job(job, nirvana, options, statface_client):
    job = job.env(
        yt_spec_defaults={
            "pool_trees": ["physical"],
            "use_default_tentative_pool_trees": True,
        },
        templates={
            "tmp_root": "//tmp/rkam/regular_img_pdb_money",
        }
    )

    output_schema_0 = {
        "cost": Float,
        "show": UInt64,
        "click": UInt64,
        "device": String,
        "page_id": String,
    }

    pages = get_pages()
    hints = with_hints(output_schema_0)

    mapper = hints(ParseCheventImg(pages))

    report = ns.StatfaceReport().path(
        "Image_RU/Others/Partner2/bschevent_pageid_1d",
    ).scale(
        "daily",
    ).client(
        statface_client,
    )

    job.table(
        "//logs/bs-chevent-log/1d/@dates"
    ).filter(
        nf.and_(
            nf.or_(
                nf.equals("placeid", "542"),
                nf.equals("placeid", "1542"),
            ),
            nf.equals("fraudbits", "0"),
        )
    ).map(
        mapper,
    ).groupby(
        "page_id",
        "device",
    ).aggregate(
        sum_show=na.sum("show"),
        sum_click=na.sum("click"),
        sum_cost=na.sum("cost"),
    ).project(
        ne.all(),
        fielddate=ne.const(options.dates[0]),
    ).publish(
        report,
        allow_change_job=True,
    )

    return job


if __name__ == '__main__':
    cli.run()
