# -*- coding: utf-8 -*-
import datetime
import logging
from links import link_functions

def run_test(test, tokens, test_functions):
    test_func = test_functions[test['type']][test['subtype']]
    status = test_func(**dict(test['params'], tokens=tokens))
    cur_timestamp = datetime.datetime.isoformat(datetime.datetime.now())
    result = {
        'test': test,
        'status': status,
        'update_time': cur_timestamp
    }
    return result


def apply_check(test_result, checkers, yt_path):
    update_time = datetime.datetime.strptime(test_result['update_time'].split('.')[0], "%Y-%m-%dT%H:%M:%S")
    cur_time = datetime.datetime.now()
    time_delta = cur_time - update_time

    test_result['yt_path'] = yt_path

    if test_result['status'] == "ERROR":
        test_result['check_status'] = "ERROR"
        return

    if time_delta.days >= test_result['test']['checker']['freshness']:
        test_result['check_status'] = "OUTDATED"
        return

    check_func = checkers[test_result['test']['checker']['type']]
    check_pass = check_func(**dict(test_result['test']['checker']['params'], status=test_result['status']))
    if check_pass:
        test_result['check_status'] = "PASS"
    else:
        test_result['check_status'] = "FAIL"


def create_html_result(test_results):
    status_color = {
        'FAIL': 'red',
        'PASS': 'green',
        'ERROR': 'maroon',
        'OUTDATED': 'blue'
    }
    sort_priority = {
        'ERROR': 1,
        'FAIL': 2,
        'OUTDATED': 3,
        'PASS': 4
    }
    sorted_results = sorted(test_results, key=lambda x: (sort_priority[x['check_status']], x['test']['name']))
    html = u"<html><body><h2>Мониторинг аналитики</h2><br/>"
    html += '<table border="1">'
    html += '<tr><td><b>{name}</b></td><td><b>{status}</b></td><td><b>{responsible}</b></td><td><b>{links}</b></td></tr>'.format(
        name="Test",
        status="Status",
        responsible="Responsible",
        links="Links"
    )

    for res in sorted_results:
        responsibles = []
        for person in res['test']['responsible']:
            responsibles.append('<a href="https://staff.yandex-team.ru/{person}">{person}</a>'.format(person=person))

        links = []
        data_for_links = res['test'].get('links', [])
        if 'link_generator' in res['test']:
            link_generator = link_functions.get(res['test']['link_generator']['type'])
            try:
                if link_generator is not None:
                    for url, name in link_generator(**dict(res['test']['link_generator']['params'], test=res['test'])):
                        data_for_links.append({'url': url, 'name': name})
                else:
                    logging.error("Link function {} not found".format(res['test']['link_generator']['type']))
            except Exception as e:
                logging.error("Link generation error for {}".format(res['test']['name']))

        for link in data_for_links:
            links.append('<a href="{url}">{name}</a>'.format(url=link['url'], name=link['name']))

        test_link = link_functions['yt_path'](res['test'], res['yt_path'])[0][0]

        html += '<tr><td><a href="{test_link}">{name}</a></td><td><font color="{color}">{status}</font></td><td>{responsible}</td><td>{links}</td></tr>'.format(
            test_link=test_link,
            name=res['test']['name'],
            color=status_color[res['check_status']],
            status=res['check_status'],
            responsible=" ".join(responsibles),
            links=" ".join(links)
        )
    html += '</table>'
    html += '</body></html>'

    return html


def log_request(r):
    logging.info('{} {}'.format(r.request.method, r.url))
    logging.info(r.request.body)
    headers_secret = dict((k, v) if k != 'Authorization' else (k, '***') for k, v in r.request.headers.items())
    logging.info(headers_secret)
    logging.info("status code {}, reason '{}'".format(r.status_code, r.reason))
