import argparse
import datetime

from nile.api.v1 import (
    filters as nf,
    aggregators as na,
    extractors as ne,
    clusters,
    Record,
)
import nile
import logging
logging.basicConfig(format='[%(asctime)s] %(filename)s[LINE:%(lineno)d] %(levelname)-8s %(message)s',
                    level=logging.ERROR)


def argument_parser():
    parser = argparse.ArgumentParser(description='Get parameters')
    parser.add_argument(
        '-p',
        dest='pool',
        type=str,
        help='you hahn pool',
    )
    parser.add_argument(
        '-r',
        dest='rpath',
        type=str,
        help='result path',
    )
    parser.add_argument(
        '-t',
        dest='timestamp',
        type=int,
        help='timestamp',
    )
    parser.add_argument(
        '-mf',
        dest='metric',
        type=str,
        help='metric file',
    )
    args = parser.parse_args()
    return args


def main():
    args = argument_parser()
    timestamp = args.timestamp
    job_root = args.rpath

    if timestamp > 100000000000:
        timestamp = timestamp / 1000

    current_day = datetime.datetime.fromtimestamp(timestamp) - datetime.timedelta(1)
    current_date = '{}-{:0>2}-{:0>2}'.format(
        current_day.year,
        current_day.month,
        current_day.day
    )

    cluster = clusters.Hahn(pool=args.pool).env(templates=dict(
        job_root=job_root,
        )
    )
    final_result = list(cluster.read('$job_root/{}'.format(current_date)))

    result_dict = {}
    result_dict_turbo = {}
    result_dict_chats = {}

    for elem in final_result:
        result_dict[elem['ui']] = {}
        result_dict[elem['ui']]['total_click'] = elem['total_click']
        result_dict[elem['ui']]['req_count'] = elem['count']

        result_dict_turbo[elem['ui']] = {}
        result_dict_turbo[elem['ui']]['total_turbo_click'] = elem['total_turbo_click']
        result_dict_turbo[elem['ui']]['misc_turbo_click'] = elem['misc_turbo_click']
        result_dict_turbo[elem['ui']]['wiki_turbo_click'] = elem['wiki_turbo_click']
        result_dict_turbo[elem['ui']]['snippet_nonwiki_turbo_click'] = elem['snippet_nonwiki_turbo_click']
        result_dict_turbo[elem['ui']]['entity_turbo_click'] = elem['entity_turbo_click']
        result_dict_turbo[elem['ui']]['news_turbo_click'] = elem['news_turbo_click']
        result_dict_turbo[elem['ui']]['instructions_turbo_click'] = elem['instructions_turbo_click']
        result_dict_turbo[elem['ui']]['req_with_turbo_click'] = elem['with_turbo_click']
        result_dict_turbo[elem['ui']]['req_with_turbo'] = elem['with_turbo']
        result_dict_turbo[elem['ui']]['req_with_wiki_turbo'] = elem['with_wiki_turbo']
        result_dict_turbo[elem['ui']]['req_with_snippet_nonwiki_turbo'] = elem['with_snippet_nonwiki_turbo']
        result_dict_turbo[elem['ui']]['req_with_entity_turbo'] = elem['with_entity_turbo']
        result_dict_turbo[elem['ui']]['req_with_news_turbo'] = elem['with_news_turbo']
        result_dict_turbo[elem['ui']]['req_with_instructions_turbo'] = elem['with_instructions_turbo']

        result_dict_chats[elem['ui']] = {}
        result_dict_chats[elem['ui']]['chats_click'] = elem['chats_click']
        result_dict_chats[elem['ui']]['req_with_chats_click'] = elem['with_chats_click']
        result_dict_chats[elem['ui']]['req_with_chats'] = elem['with_chats']

    result_dict['all_platforms'] = {}
    result_dict_turbo['all_platforms'] = {}
    result_dict_chats['all_platforms'] = {}

    for key in result_dict_turbo['touch']:
        result_dict_turbo['all_platforms'][key] = sum(result_dict_turbo[ui][key] for ui in ['touch', 'app', 'pad', 'desktop'])

    for key in result_dict_chats['touch']:
        result_dict_chats['all_platforms'][key] = sum(result_dict_chats[ui][key] for ui in ['touch', 'app', 'pad', 'desktop'])

    for key in result_dict['touch']:
        result_dict['all_platforms'][key] = sum(result_dict[ui][key] for ui in ['touch', 'app', 'pad', 'desktop'])

    result_dict_turbo['touch_and_app'] = {}
    result_dict['touch_and_app'] = {}
    for key in result_dict_turbo['touch']:
        result_dict_turbo['touch_and_app'][key] = sum(result_dict_turbo[ui][key] for ui in ['touch', 'app'])
    for key in result_dict['touch']:
        result_dict['touch_and_app'][key] = sum(result_dict[ui][key] for ui in ['touch', 'app'])

    with open(args.metric, 'w') as metric_file:
        for ui in ['touch', 'app', 'touch_and_app']:
            for elem in result_dict_turbo[ui]:
                metric_file.write('{}|turbo|{}\t{}\t{}\n'.format(
                    ui,
                    elem,
                    result_dict_turbo[ui][elem],
                    timestamp-60*60*24
                ))
            metric_file.write('{}|turbo|total_turbo_click_freq\t{}\t{}\n'.format(
                ui,
                float(result_dict_turbo[ui]['total_turbo_click']) / result_dict[ui]['total_click'],
                timestamp-60*60*24
            ))
        for ui in ['touch', 'app', 'touch_and_app', 'desktop', 'pad', 'all_platforms']:
            for elem in result_dict[ui]:
                metric_file.write('{}|{}\t{}\t{}\n'.format(
                    ui,
                    elem,
                    result_dict[ui][elem],
                    timestamp-60*60*24
                ))
        for ui in ['touch', 'app', 'desktop', 'pad', 'all_platforms']:
            for elem in result_dict_chats[ui]:
                metric_file.write('{}|chats|{}\t{}\t{}\n'.format(
                    ui,
                    elem,
                    result_dict_chats[ui][elem],
                    timestamp-60*60*24
                ))

if __name__ == "__main__":
    main()
