# coding: utf-8

import json
import hashlib

from jinja2 import Template
import library.python.resource as resource
import library.python.svn_version as sv

from nile.nodes.operation import Publish
from nile.nodes.table import Table, RemoteOutputTable, RemoteInputTable, VoidOutputTable

from utils import FLOW_NAME_SEPARATOR


GROUPS = []


def get_group_id(label):
    class_name = label.split('.')[0]
    if class_name not in GROUPS:
        GROUPS.append(class_name)
    return GROUPS.index(class_name)


def is_output_table(node_object):
    return isinstance(node_object, RemoteOutputTable)


def get_report_path(node_object):
    return str(node_object.report._path)


def get_table_path(node_object):
    return str(node_object.path)


def get_node_label(node):
    if isinstance(node.object, Publish):
        return get_report_path(node.object)
    if isinstance(node.object, Table) and not isinstance(node.object, VoidOutputTable):
        return get_table_path(node.object)

    for edge in node.output_edges:
        if edge.data.get('label') is not None:
            return edge.data.get('label')
    return None


def format_stat_label(label):
    return label.split('/')[-1]


def format_node_label(label):
    return '\\n'.join(label.split('.'))


def is_good_node(node):
    # Возвращает конечные узлы дерева или операции, у которых есть label
    if node is None:
        return False
    return get_node_label(node) is not None


def get_next_good_node(node):
    if is_good_node(node):
        yield node
    else:
        for child in node.outputs:
            for good_node in get_next_good_node(child):
                yield good_node


def get_node_hash(label):
    return hashlib.md5(str(label)).hexdigest()


def get_html_template():
    return Template(resource.find('templates/template.html').decode('utf-8'))


def format_digraph(flow_graph, output_tables):  # noqa
    used_nodes = set()

    # преобразовать граф:
    # * удалить крайние VoidOutputTable
    # * перед RemoteOutputTable и Table вставить свою операцию
    for edge in flow_graph.edges:
        from_ = edge.from_
        to_ = edge.to_
        from_node_label = get_node_label(from_)
        if isinstance(from_.object, RemoteOutputTable) and isinstance(to_.object, VoidOutputTable):
            edge.remove()
            to_.remove()

            output_fn_name = output_tables[from_node_label][2]
            new_node = flow_graph.add_node(VoidOutputTable())
            from_.redirect_input_edges_to(new_node, remove_originals=True)
            flow_graph.add_edge(new_node, from_, output=0, input=0, data=dict(label=output_fn_name))

    for edge in flow_graph.edges:
        from_ = edge.from_
        to_ = edge.to_
        to_node_label = get_node_label(to_)
        from_node_label = get_node_label(from_)

        if is_good_node(to_) and to_.id not in used_nodes:
            if isinstance(to_.object, RemoteOutputTable):
                yield '{target} [label="{label}" shape="image" image="{image_url}"]'.format(
                    target=get_node_hash(to_node_label),
                    label=to_node_label,
                    image_url='yt_output_table',
                )
            elif isinstance(to_.object, Publish):
                yield '{target} [label="{label}" shape="image" image="{image_url}"]'.format(
                    target=get_node_hash(to_node_label),
                    label=format_stat_label(to_node_label),
                    image_url='stat_report',
                )
            else:
                yield '{target} [label="{label}" shape="box" borderWidth="{borderWidth}" group="{group}"]'.format(
                    target=get_node_hash(to_node_label),
                    label=format_node_label(to_node_label),
                    group=get_group_id(to_node_label),
                    borderWidth=2 if to_.out_degree > 1 else 1,
                )
            used_nodes.add(to_.id)

        if is_good_node(from_) and from_.id not in used_nodes:
            if isinstance(from_.object, RemoteInputTable):
                yield '{source} [label="{label}" shape="image" image="{image_url}" ]'.format(
                    source=get_node_hash(from_node_label),
                    label=from_node_label,
                    image_url='yt_input_table',
                )
            else:
                yield '{source} [label="{label}" shape="box" borderWidth="{borderWidth}" group="{group}"]'.format(
                    source=get_node_hash(from_node_label),
                    label=format_node_label(from_node_label),
                    group=get_group_id(from_node_label),
                    borderWidth=2 if from_.out_degree > 1 else 1,
                )
            used_nodes.add(from_.id)

        if is_good_node(from_):
            if is_good_node(to_):
                yield '{source} -> {target}'.format(
                    source=get_node_hash(from_node_label),
                    target=get_node_hash(to_node_label),
                )
            else:
                for new_to_ in get_next_good_node(to_):
                    yield '{source} -> {target}'.format(
                        source=get_node_hash(from_node_label),
                        target=get_node_hash(get_node_label(new_to_)),
                    )


def get_code_url(fn):
    return 'https://a.yandex-team.ru/arc/trunk/arcadia/{}?rev={}#L{}'.format(
        fn.func_code.co_filename,
        sv.svn_revision(),
        fn.func_code.co_firstlineno,
    )


def get_nodes_data(flow_graph, output_tables, nodes, cluster):
    data = {}
    for node in flow_graph.sort():
        node_label = get_node_label(node)
        if node_label is None:
            continue
        node_label_main = node_label.split(FLOW_NAME_SEPARATOR)[0]
        node_id = get_node_hash(node_label)
        if node_id in data:
            continue

        data[node_id] = {}

        if isinstance(node.object, Publish):
            data[node_id] = {
                'type': 'Publish',
                'label': node_label,
                'url': 'https://stat.yandex-team.ru/' + get_report_path(node.object),
                'text': '',
            }
        elif isinstance(node.object, Table) and not isinstance(node.object, VoidOutputTable):
            table_path = get_table_path(node.object)
            data[node_id] = {
                'type': 'Table',
                'label': node_label,
                'url': 'https://yt.yandex-team.ru/{}/navigation?path={}'.format(cluster, table_path),
                'text': '',
            }
        else:
            fn = nodes[node_label_main][1]
            data[node_id] = {
                'type': 'Operation',
                'label': node_label,
                'url': get_code_url(fn),
            }
            if fn.__doc__:
                data[node_id]['text'] = fn.__doc__.decode('utf-8')

        for output_table, (plot_class, plot_fn, label) in output_tables.items():
            # для выходных таблиц, которые сформированы генератором внутри одного узла
            label_main = label.split(FLOW_NAME_SEPARATOR)[0]
            node_id = get_node_hash(label)
            main_node_id = get_node_hash(label_main)
            if node_id in data or main_node_id not in data:
                continue

            data[node_id] = data[main_node_id]

    return data


def get_plotter_vis_html(plotter):
    flow_graph = plotter.job.flow_graph
    output_tables = plotter.output_tables
    nodes = plotter.nodes

    html = get_html_template()
    digraph = '\n'.join(list(sorted(format_digraph(flow_graph, output_tables), reverse=True)))

    if plotter.require_plots:
        viz_type = 'plots: {}'.format(plotter.require_plots)
    else:
        viz_type = 'layers: {}'.format(plotter.layers)

    return html.render(
        digraph=digraph,
        nodes_data=json.dumps(get_nodes_data(flow_graph, output_tables, nodes, plotter.cluster), ensure_ascii=False),
        viz_type=viz_type,
        FLOW_NAME_SEPARATOR=FLOW_NAME_SEPARATOR,
    )
