import sys, json, codecs
from PIL import Image, ImageStat
from functools import partial
from multiprocessing import Pool

def grayscale_max(img, pixels):
    if len(pixels) == 0:
        return 0.0
    return max(map(sum, pixels))


def grayscale_avg(img, pixels):
    if len(pixels) == 0:
        return 0.0
    return sum(map(sum, pixels)) / len(pixels)


def grayscale_nth_stat(img, pixels, n):
    if len(pixels) == 0:
        return 0.0

    pos = int((len(pixels) - 1) / 100.0 * n)
    return sum(pixels[pos])


def map_max_scale(num):
    if num < 1000:
        return 0.0
    if num < 2000:
        return 0.1
    if num < 3000:
        return 0.2
    if num < 4000:
        return 0.4
    if num < 5000:
        return 0.6
    if num < 10000:
        return 0.8
    return 1.0


def map_avg_scale(num):
    if num < 100:
        return 0.0
    if num < 150:
        return 0.1
    if num < 250:
        return 0.2
    if num < 400:
        return 0.4
    if num < 600:
        return 0.6
    if num < 1000:
        return 0.8
    return 1.0


def cut_image_border(img, percent):
    if percent < 0.1:
        return img
    x, y = img.size
    x_min = int(x / 100.0 * percent)
    y_min = int(y / 100.0 * percent)
    return img.crop((x_min, y_min, x - x_min, y - y_min))


def process_image(img):
    try:
        pil_img = Image.open(img['filename'])
    except (IOError, ValueError) as e:
       return {
            'url': img['url'],
            'filename': img['filename'],
            'errors': 'Could not read image'
        }

    try:
        pil_img = pil_img.convert('RGB')
    except IOError:
        return {
            'url': img['url'],
            'filename': img['filename'],
            'errors': 'Could not convert image to RGB'
        }

    result = {'image_avatars_url': img['url']}
    for border_size in [0, 15, 25]:
        bordered = cut_image_border(pil_img, border_size)
        for thumb_size in [20, 40, 80]:
            thumb = bordered.resize((thumb_size, thumb_size), Image.BICUBIC)

            pixels = []
            for pixel in thumb.getdata():
                mu = sum(pixel) / float(3)
                pixels.append([(pixel[i] - mu) * (pixel[i] - mu) for i in [0, 1, 2]])

            pixels.sort(key=lambda x: sum(x))

            for func, func_name in [(grayscale_avg, 'avg'),
                                    (grayscale_max, 'max'),
                                    (partial(grayscale_nth_stat, n=50), 'nth50'),
                                    (partial(grayscale_nth_stat, n=75), 'nth75'),
                                    (partial(grayscale_nth_stat, n=90), 'nth90')
                                    ]:
                scale_name = 'grayscale_{}_b{}_t{}'.format(func_name, border_size, thumb_size)
                try:
                    scale = func(thumb, pixels)
                    if func_name == 'avg':
                        result[scale_name] = map_avg_scale(scale)
                    else:
                        result[scale_name] = map_max_scale(scale)
                except Exception as e:
                    continue

    return result


def main():
    in_file = sys.argv[1]
    n_proc = int(sys.argv[2])
    out_file = sys.argv[3]

    with codecs.open(in_file, 'r', 'utf8') as f:
        in_data = json.load(f, encoding='utf8')

    pool = Pool(processes=n_proc)
    out_data = pool.map(process_image, in_data)
    out_data = [ x for x in out_data if 'errors' not in x]

    with codecs.open(out_file, 'w', 'utf8') as f:
        json.dump(out_data, f, ensure_ascii=False, indent=2)


if __name__ == '__main__':
    if len(sys.argv) != 4:
        print >>sys.stderr, "Usage:", sys.argv[0], "<in_file> <n_proc> <out_file>"
        sys.exit(1)
    main()
