import sys, json, codecs
from PIL import Image, ImageStat
from functools import partial
from multiprocessing import Pool

def grayscale_max(img, thumb_size=40):
    thumb = img.resize((thumb_size, thumb_size), Image.BICUBIC)
    current_max = 0
    for pixel in thumb.getdata():
        mu = sum(pixel) / float(3)
        current_max = max(sum((pixel[i] - mu) * (pixel[i] - mu) for i in [0, 1, 2]), current_max)
    return current_max


def grayscale_avg(img, thumb_size=40):
    thumb = img.resize((thumb_size, thumb_size), Image.BICUBIC)
    s = 0
    cnt = 0
    for pixel in thumb.getdata():
        mu = sum(pixel) / float(3)
        s += sum((pixel[i] - mu) * (pixel[i] - mu) for i in [0, 1, 2])
        cnt += 1
    if cnt == 0:
        return 0
    return float(s) / cnt


def process_image(args):
    img, func = args
    try:
        pil_img = Image.open(img['filename'])
    except IOError:
        return {
            'url': img['url'],
            'filename': img['filename'],
            'errors': 'Could not read image'
        }
    except ValueError:
        return {
            'url': img['url'],
            'filename': img['filename'],
            'errors': 'Could not read image'
        }

    try:
        pil_img = pil_img.convert('RGB')
    except IOError:
        return {
            'url': img['url'],
            'filename': img['filename'],
            'errors': 'Could not convert image to RGB'
        }

    try:
        max_gray_deviation = func(pil_img)
    except Exception as e:
        return {
            'url': img['url'],
            'filename': img['filename'],
            'errors': 'Grayscale func error for {}: {}'.format(img['url'], str(e))
        }

    return {
        'url': img['url'],
        'filename': img['filename'],
        'max_gray_deviation': max_gray_deviation,
    }


def main():
    in_file = sys.argv[1]
    n_proc = int(sys.argv[2])
    func_name = sys.argv[3]
    out_file = sys.argv[4]

    if func_name == 'max':
        grayscale_func = grayscale_max
    elif func_name.startswith('max_'):
        grayscale_func = partial(grayscale_max, thumb_size=int(func_name.split('_')[-1]))
    elif func_name == 'avg':
        grayscale_func = grayscale_avg
    elif func_name.startswith('avg_'):
        grayscale_func = partial(grayscale_avg, thumb_size=int(func_name.split('_')[-1]))
    else:
        print >>sys.stderr, "Unknown func", func_name
        sys.exit(1)

    with codecs.open(in_file, 'r', 'utf8') as f:
        in_data = json.load(f, encoding='utf8')

    pool = Pool(processes=n_proc)
    out_data = pool.map(process_image, [(img, grayscale_func) for img in in_data])

    with codecs.open(out_file, 'w', 'utf8') as f:
        json.dump(out_data, f, ensure_ascii=False, indent=2)


if __name__ == '__main__':
    if len(sys.argv) != 5:
        print >>sys.stderr, "Usage:", sys.argv[0], "<in_file> <n_proc> <func: max|avg> <out_file>"
        sys.exit(1)
    main()
