import sys
import argparse
import json
import time
import logging
import codecs
from toloka_api_request import custom_toloka_request

def str2bool(v):
    if v.lower() in ('yes', 'true', 't', 'y', '1'):
        return True
    elif v.lower() in ('no', 'false', 'f', 'n', '0'):
        return False
    else:
        raise argparse.ArgumentTypeError('Boolean value expected.')


def create_argument_parser():
    parser = argparse.ArgumentParser()

    parser.add_argument('--token', required=True)
    parser.add_argument('--environment', required=True)
    parser.add_argument('--pool-id-field', default='poolId')
    parser.add_argument('--sleep-time', type=int, default=5)
    parser.add_argument('--tasks-limit', type=int, default=100000)

    parser.add_argument('--awaited-honeypots', type=int, required=True)
    parser.add_argument('--awaited-hints', type=int, required=True)
    parser.add_argument('--awaited-total', type=int, default=0)
    parser.add_argument('--accept-manual-close', type=str2bool, nargs='?', default=False, const=True)

    parser.add_argument('--input', required=True)
    parser.add_argument('--output-tasks', required=True)
    parser.add_argument('--output-honeypots', required=True)
    parser.add_argument('--output-hints', required=True)

    return parser


def get_tasks(pool_id, token, tasks_limit, sandbox):
    response = custom_toloka_request('tasks', 'GET', token,
                        {'pool_id': pool_id, 'limit': tasks_limit}, sandbox)
    return response['items']


def pool_is_manually_closed(pool_id, token, sandbox):
    status = custom_toloka_request('pools/{}'.format(pool_id), 'GET', token, {}, sandbox)
    if status['status'] == 'CLOSED' and 'last_stopped' in status and \
            status.get('last_close_reason', '') == 'MANUAL':
        return True
    return False


def wait_tasks_from_pool(pool_id, environment, token, sleep_time, tasks_limit, \
                        awaited_hints, awaited_honeypots, awaited_total, accept_manual_close):
    logging.info('Starting tasks wait loop')

    if environment.upper() == 'PRODUCTION':
        sandbox = False
    else:
        sandbox = True

    logging.info('Waiting for: {} hints, {} honeypots, {} hints+honeypots'.format(\
                        awaited_hints, awaited_honeypots, awaited_total))

    while True:
        hints, honeypots, tasks = [], [], []

        unfiltered_tasks = get_tasks(pool_id, token, tasks_limit, sandbox)
        for t in unfiltered_tasks:
            if 'message_on_unknown_solution' in t and 'known_solutions' in t:
                hints.append(t)
            elif 'known_solutions' in t:
                honeypots.append(t)
            else:
                tasks.append(t)

        n_hints = len(hints)
        n_honeypots = len(honeypots)
        n_tasks = len(tasks)
        logging.info('Stats: {} hints, {} honeypots, {} hints+honeneypots'.format(\
                                n_hints, n_honeypots, n_hints + n_honeypots))

        if accept_manual_close and pool_is_manually_closed(pool_id, token, sandbox):
            logging.info('Stopped because of manual closing')
            break

        if n_hints >= awaited_hints and n_honeypots >= awaited_honeypots and \
                        n_hints + n_honeypots >= awaited_total:
            break

        time.sleep(sleep_time)

    logging.info('Tasks wait loop is finished')
    return hints, honeypots, tasks


def main():
    logging.basicConfig(format='%(asctime)s - %(name)s - %(levelname)s - %(message)s',
             level=logging.INFO)
    logging.info('Started')

    args = create_argument_parser().parse_args()
    logging.info('Sleep time {}'.format(args.sleep_time))

    with codecs.open(args.input, 'r', 'utf8') as f:
        in_data = json.load(f, encoding='utf8')
        logging.info('Input file is loaded')

    if len(in_data) == 0:
        hints, honeypots, tasks = [], [], []
    else:
        hints, honeypots, tasks = wait_tasks_from_pool( in_data[0][args.pool_id_field],
                                                        args.environment,
                                                        args.token,
                                                        60 * args.sleep_time,
                                                        args.tasks_limit,
                                                        args.awaited_hints,
                                                        args.awaited_honeypots,
                                                        args.awaited_total,
                                                        args.accept_manual_close
                                                        )

    with codecs.open(args.output_hints, 'w', 'utf8') as f:
        json.dump(hints, f, indent=4, ensure_ascii=False)
        logging.info('Hints are saved')

    with codecs.open(args.output_honeypots, 'w', 'utf8') as f:
        json.dump(honeypots, f, indent=4, ensure_ascii=False)
        logging.info('Honeypots are saved')

    with codecs.open(args.output_tasks, 'w', 'utf8') as f:
        json.dump(tasks, f, indent=4, ensure_ascii=False)
        logging.info('Tasks are saved')


if __name__ == '__main__':
    main()
