PRAGMA AnsiInForEmptyOrNullableItemsCollections;

$script = @@
import collections

EARNING_PERIOD = 5 * 60  # 5 minutes

def create_row(rec, revenue):
    return dict(
        day=rec.day,
        revenue=revenue,
        start_ts=rec.start_ts,
        user_id=rec.user_id,
    )

def enrich_money(key, recs):
    prev_rec = None
    tot_revenue = collections.defaultdict(float)
    earning_end_ts = 0

    for rec in recs:
        if rec.priority == 0:
            if prev_rec:
                yield create_row(prev_rec, tot_revenue or None)

            prev_rec = rec
            tot_revenue = collections.defaultdict(float)
            earning_end_ts = rec.start_ts + rec.duration + EARNING_PERIOD

        elif rec.start_ts < earning_end_ts:
            tot_revenue[rec.service] += rec.revenue

    if prev_rec:
        yield create_row(prev_rec, tot_revenue or None)
@@;

$enrich_money = Python3::enrich_money(
    Callable<(
        Tuple<String?, String?>,
        Stream<
            Struct<
                'day':String?,
                'user_id':String?,
                'start_ts':UInt64?,
                'duration':Uint32?,
                'service':String?,
                'revenue':Double?,
                'priority':Int32?,
            >
        >
    ) -> Stream<
        Struct<
            'day':String,
            'user_id':String,
            'start_ts':UInt64,
            'revenue':Dict<String,Double>?,
        >
    >>,
    $script
);

DEFINE SUBQUERY $enrich_with_money($stream, $start_date, $end_date, $cluster) AS
    $stream_with_money = (
        select
            s.*,
            0 as priority,
        from $stream() as s
        where s.yandexuid is not null

        union all

        select
            TableName(TablePath(), "yt") as day,
            revenue_by_eventtime as revenue,
            service,
            yandexuid,
            1 AS priority,
            cast(`timestamp` as UInt64) as start_ts,
        from yt:$cluster.range("//statbox/cube/daily/event_money/v2.1", $start_date, $end_date)
        where
            revenue_by_eventtime != 0
            and yandexuid is not null
    );

    $money_info = (
        reduce $stream_with_money
        presort start_ts, priority
        on day, yandexuid
        using $enrich_money(TableRow())
    );

    select
        s.*,
        mi.revenue as revenue,
    from $stream() as s
    left join any $money_info as mi
    using (day, user_id, start_ts)
END DEFINE;

DEFINE SUBQUERY $_add_yuid_to_spylog($stream, $start_date, $end_date, $cluster) AS
    $i2y = (
        select *
        from yt:$cluster.range("//home/searchshare/squeeze/icookie_to_yuid", $start_date, $end_date)
        with schema Struct<day:String,user_id:String,yandexuid:String>
    );

    select
        s.*,
        substring(s.user_id, 1) AS yandexuid,
    from $stream() AS s
    where s.ui = "desktop"

    union all

    select
        s.*,
        i2y.yandexuid AS yandexuid,
    from $stream() AS s
    left join any $i2y as i2y
    using (day, user_id)
    where s.ui = "touch";
END DEFINE;

$add_yuid_to_spylog = ($stream, $start_date, $end_date, $cluster) -> (
    ($world) -> (
        $_add_yuid_to_spylog($world, $stream, $start_date, $end_date, $cluster)
    )
);

export $enrich_with_money, $add_yuid_to_spylog;
