# -*- coding: utf-8 -*-

import os
import random
import re

from decorators import render_to, jsonify, user_has_permissions
from models import Status, Query
from django.core.exceptions import ObjectDoesNotExist
from collections import namedtuple

import yt.wrapper as yt

VIEWERS_DIR = '//home/search-functionality/serp_anatomy/queries_viewer'

QueriesGroup = namedtuple('QueriesGroup', 'ui, viewer, queries, width, qwidth, qheight')


# for development and testing
def fake_queries(rows_cnt, cols_cnt):
    return map(str, range(rows_cnt * cols_cnt))


def parse_viewer_info(viewer_raw):
    viewer, size = viewer_raw.split(':')
    rows_cnt, cols_cnt = size.split('x')
    return viewer, int(rows_cnt), int(cols_cnt)


def get_queries(rows_cnt, cols_cnt, status):
    table = VIEWERS_DIR + '/' + status.viewer
    yt_client = yt.YtClient(proxy='hahn', token=os.environ['YT_TOKEN'])
    modification_time = yt.get(table + '/@modification_time', client=yt_client)
    if status.modification_time != modification_time:
        Query.objects.filter(viewer=status.viewer).delete()
        for ind, row in enumerate(yt.read_table(table, client=yt_client)):
            Query.objects.create(query=row['query'], index=ind, viewer=status.viewer)
        status.modification_time = modification_time
        status.save()

    count = rows_cnt * cols_cnt
    offset = status.offset
    queries_objects = Query.objects.filter(viewer=status.viewer)
    count_all = queries_objects.count()
    if count_all <= count:
        queries = queries_objects
    else:
        queries = list(queries_objects[offset:offset + count])
        if len(queries) < count:
            queries.extend(queries_objects[:count - len(queries)])

    status.offset = (offset + count) % count_all
    status.save()
    return map(lambda x: x.query, queries)


@user_has_permissions()
@render_to(t='queries_viewer.html')
def index(request, userinfo):
    viewers_raw = request.GET.get('viewers')
    if viewers_raw is None:
        raise AttributeError('Url param "viewers" was\'t found')

    refresh = {}
    if request.GET.get('refresh'):
        url = request.build_absolute_uri()
        url = re.sub(r'&rand=[\d\.]+', '', url)
        url += '&rand=%d' % random.randint(1, 10000)
        refresh['seconds'] = request.GET['refresh']
        refresh['url'] = url

    ui = request.GET.get('ui')
    if not ui or ui == 'touch':
        ui = 'touch/'
    elif ui == 'desktop':
        ui = ''
    elif ui == 'pad':
        ui = 'pad/'
    else:
        raise ValueError('ui parameter can be one of [touch, desktop, pad]')

    test_id = request.GET.get('test-id')

    try:
        viewers_info = map(parse_viewer_info, viewers_raw.split(','))
    except ValueError:
        raise ValueError(
            'Viewers should be formatted as {viewer_name}:{rows_count}x{columns_count} and separated by \',\'')
    cols_cnt_all = sum(x[2] for x in viewers_info)
    groups = []
    for viewer, rows_cnt, cols_cnt in viewers_info:
        try:
            status = Status.objects.get(viewer=viewer)
        except ObjectDoesNotExist:
            status = Status(viewer=viewer, offset=0, modification_time=None)
        queries = get_queries(rows_cnt, cols_cnt, status)
        groups.append(QueriesGroup(
            ui,
            viewer,
            queries,
            100.0 / cols_cnt_all * cols_cnt,
            100.0 / cols_cnt,
            100.0 / rows_cnt,
        ))

    return {
        'groups': groups,
        'refresh': refresh,
        'test_id': test_id,
    }
