# coding=utf-8
import re

__author__ = 'irlab'


class SerpSettings:
    RESOLUTION_DESKTOP = "683;1020"
    RESOLUTION_TOUCH = "360;567"

    def __init__(self):
        self.MAIN_DIV_ID = None

        # проход по dom-дереву сверху-вниз - ищем "элементы серпа",
        # рекурсивно разбиваем узлы на составные элементы анатомии, пока не наткнемся на узел, удовлетворяющий двум условиям:
        #   1) тег в списке TAG_NAMES_NOT_TO_SKIP_DOWN
        #   2) тег не удовлетворяет ни одному из селекторов SELECTORS_TO_SKIP_DOWN
        self.TAG_NAMES_NOT_TO_SKIP_DOWN = None

        self.SELECTORS_TO_SKIP_DOWN = None

        # выделяем из dom-дерева элемента серпа его "хребет" - то есть все теги, кроме тех, которые надо выкидывать
        # список селекторов тегов, которые надо выкидывать, задаем тут
        self.SKELETON_SKIP_SELECTORS = None
        self.SKELETON_SKIP_SELECTORS_REGEX = []
        self.SKELETON_SKIP_SELECTORS_NOBORDER = False
        self.SKELETON_USE_TAGSET = False

        # последовательность одинаковых тегов скелета длиной 3 и больше укаорачиваем до 3
        self.SKELETON_CUT_SEQUENCES_LENGTH = 2

        self.HTM_SERP_ELEMENTS_SAMPLE_COUNT = 5
        self.HTM_SERP_ELEMENTS_SKIP_IF_QUERIES_LE = 0

        self.HTM_SERP_ELEMENT_IMAGE_DIMENSION_MIN = 30

        self.RESOLUTION = self.RESOLUTION_DESKTOP

    def cleanup_html(self, html):
        return html

    def get_query_text(self, soup):
        return None

    def skeleton_accepts_class(self, css_class):
        return True

    @staticmethod
    def is_cbk(s):
        # bad class names like "0CLwBEBpqFQo"
        if re.search(r'\d', s) and re.search(r'[A-Z]', s) and re.search(r'[a-z]', s):
            return True
        elif len(re.findall(r'\d', s)) >= 5:
            return True
        return False

class GoogleSerpSettings(SerpSettings):
    def __init__(self):
        SerpSettings.__init__(self)

        self.MAIN_DIV_ID = "main"

        # проход по dom-дереву сверху-вниз - ищем "элементы серпа",
        # рекурсивно разбиваем узлы на составные элементы анатомии, пока не наткнемся на узел, удовлетворяющий двум условиям:
        #   1) тег в списке TAG_NAMES_NOT_TO_SKIP_DOWN
        #   2) тег не удовлетворяет ни одному из селекторов SELECTORS_TO_SKIP_DOWN
        self.TAG_NAMES_NOT_TO_SKIP_DOWN = ['div', 'ol', 'li', 'ul']

        self.SELECTORS_TO_SKIP_DOWN = set("""
            #main .col .mw #cnt #rcnt #center_col #res #rso
            #search #ires
            #tvcap #taw #bottomads #mbEnd
            #rhs #rhs_block
            kp-wholepage
            """.strip().split())

        # removed:
        # .srg
        # .c
        # .g
        # .kp-blk
        # .xpdopen
        # .xpdbox

        # выделяем из dom-дерева элемента серпа его "хребет" - то есть все теги, кроме тех, которые надо выкидывать
        # список селекторов тегов, которые надо выкидывать, задаем тут
        self.SKELETON_SKIP_SELECTORS = set("""
            b em span a cite style script br i svg
            .klitem-tr
            """.strip().split())

        self.SKELETON_SKIP_SELECTORS_REGEX = map(lambda x: re.compile("^" + x + "$"), [
            r'#cra-\d+-filled',   # cra-3-filled cra-29-filled
            r'\.duf\d+-\d+-\d+',  # duf3-2-18 duf3-2-19 duf3-1-0
            r'\.iukp\d+',         # iukp22 iukp23
            r'\.rhsg\d+',         # rhsg3
            r'\.rhsl\d+',         # rhsl3
            r'#uid_\d+',          # uid_18
            r'#nmtbi\d+.*',       # nmtbi11 nmtbi11cntw nmtbi11cnt
            r'#lr-sc-pts-\d+.*',
        ])
        self.SKELETON_USE_TAGSET = True

        # @TODO
        # color diff with previous

    def cleanup_html(self, html):
        # html = re.sub(r'(?si)(<script.*?>)', r'\1<!--', html)
        # html = html.replace('</script>', '--></script>')
        # html = html.replace('<script', '<!--script')
        # html = html.replace('</script>', '</script-->')
        # html = re.sub(r'(?si)(<style.*?>)', r'\1<!--', html)
        # html = html.replace('</style>', '--!></style>')
        # html = html.replace(':url(\'//', ':xurl(\'//')
        # html = html.replace(':url(//', ':xurl(//')
        # html = html.replace(':url("//', ':xurl("//')
        # html = re.sub(r'<link\s[^>*]\brel="shortcut icon">', '', html)
        html = html.replace('<head>', '<head>\n    <meta http-equiv="Content-Type" content="text/html; charset=UTF-8">')
        return html

    def get_query_text(self, soup):
        form = soup.find(name="form", attrs={ "role" : "search" })
        if form:
            query_input = form.find(name="input", attrs={ "name" : "q" })
            if query_input:
                return query_input.attrs.get("value")
        query_input = soup.find(id="lst-ib")
        if query_input:
            return query_input["value"]
        else:
            return "NOQUERY"

    def skeleton_accepts_class(self, css_class):
        if not css_class:
            return False
        if len(css_class) > 5 and re.search(r'[A-Z]', css_class) and re.search(r'[a-z]', css_class):
            # bad class names like "iDPqiVEJMenE"
            return False
        if len(css_class) > 8 and len(re.findall(r'\d', css_class)) >= 3:
            # bad class names like "r-inpj8o9qy9wo"
            return False
        if css_class.startswith('_'):
            return False
        if '.' + css_class in self.SKELETON_SKIP_SELECTORS:
            return False
        for regex in self.SKELETON_SKIP_SELECTORS_REGEX:
            if re.match(regex, '.' + css_class):
                return False

        return True


class GoogleSerpSettingsTouch(GoogleSerpSettings):
    def __init__(self):
        GoogleSerpSettings.__init__(self)

        self.RESOLUTION = self.RESOLUTION_TOUCH

        self.SELECTORS_TO_SKIP_DOWN.update(set("""
            .qs-ir .qs-io .qs-ii
            .qxproot
            .kp-body
            ---.kp-header
            ---.kp-wholepage
            #wp-tabs-container
            sticky-header
            [jsslot]
            """.strip().split()))

        self.SKELETON_SKIP_SELECTORS.update(set("""
            .nmi .nmiw .mslg .sld .vsc
            """.strip().split()))

        self.SKELETON_SKIP_SELECTORS_REGEX += map(lambda x: re.compile("^" + x + "$"), [
            r'#aig-np-.+',   # aig-np-com.google.android.apps.chromecast.app
            r'.luuv_0x.+',   # luuv_0x487611cdb9bb28e1:0x24bf5c038919f0ee
            r'#akp_uid_\d+', # akp_uid_11
            r'#i35\d+',      # i3590
        ])

        self.SKELETON_SKIP_SELECTORS_NOBORDER = True


class GoogleSerpSettingsTouchShort(GoogleSerpSettingsTouch):
    def __init__(self):
        GoogleSerpSettingsTouch.__init__(self)

        self.SKELETON_SKIP_SELECTORS = set("""
            b em span pre sup button a cite style script br i svg
            kp-carousel
            g-bubble
            g-card
            g-loading-icon
            g-review-stars
            g-scrolling-carousel
            g-white-loading-icon
            g-fab
            g-immersive-footer
            g-accordion
            g-accordion-expander
            g-tray-header
            g-sticky-content-container
            g-raised-button
            g-more-link
            date-picker-quantum
            jsl
            li
            ul
            .klitem-tr
            """.strip().split())

        self.SKELETON_SKIP_SELECTORS_REGEX = map(lambda x: re.compile("^" + x + "$"), [
            r'#cra-\d+-filled',   # cra-3-filled cra-29-filled
            r'\.duf\d+-\d+-\d+',  # duf3-2-18 duf3-2-19 duf3-1-0
            r'\.iukp\d+',         # iukp22 iukp23
            r'\.rhsg\d+',         # rhsg3
            r'\.rhsl\d+',         # rhsl3
            r'#uid_\d+',          # uid_18
            r'#nmtbi\d+.*',       # nmtbi11 nmtbi11cntw nmtbi11cnt
            r'#lr-sc-pts-\d+.*',
            r'\.r-.*',
            r'\.rl.*',
            r'\.tdu-.*',
            r'\.active',          # may be move it to SKIP_SELECTORS
            r'\.car-image',
            r'\.card-section',
            r'\.pivots-header',
            r'\.commercial-unit-mobile-.*',
            r'\.xpd.*',
            r'\.fp-w',
            r'\.gws-.*',
            r'\.kno-.*',
            r'\.husdp-.*',
            r'\.lujscdp-.*',
            r'\.lumib-.*',
            r'\.qs-.*',
            r'\.slct-col',
            r'\.ellip',
            r'\.flt-.*',
            r'\.ads-.*',
            r'\.kp-.*',
            r'\.mfr',
            r'\.mod',
            r'\.qxsd',
            r'\.g',
            r'\.g-.*',
            r'\.plalb-.*',
            r'\.y',
            r'\.yi',
            r'\.yp',
            r'\.see-more-unit',
            r'\.vk_.*',
            r'\.tab',
            r'\.threecol',
            r'\.lr_.*',
            r'\.lubh-.*',
            r'\.sae-.*',
            r'\.dir-tt-.*',
            r'\.notice-.*',
            r'\.numbered-step.*',
            r'\.lbdh',
            r'\.qxp.*' # or qxproot only ?
        ])
