import math

# tags that do not block a new session
params_block_intersection_set = frozenset(("cmnt", "touch", "desktop"))
params_split_by_tags = True
params_transition_new_session = True
# social transition penalty
params_st_penalty = 60
# first blue point bonus
params_first_blue_bonus = True
params_session_length = 1800
params_ignore_zen = False
# full length of colored point intervals
params_red_interval = 60
params_blue_interval = 60
# social transition bonus
params_transition_bonus = False
# legacy parameter for transition duplicates
params_down_time = 0
# decay coefficient for half-social transitions
params_black_transition_decay = 0.995


def new_session_criterion(tags, new_tags):
    """checks if there is a meaningful intersection between session tags and a new point"""
    intersection = (set(tags.keys()) & set(new_tags.keys())) - params_block_intersection_set
    return len(intersection) == 0


def calc_sessions(list_of_points):
    color_interval = {"red": params_red_interval, "blue": params_blue_interval, "black": 0}
    tags = dict()
    penalized_tags = set()
    session_ts, end_ts, color_ts, prev_ts, has_push = None, None, None, None, False
    bonus, penalty, bonus_all, last_l, last_r = 0, 0, 0, 0, 0

    # bonus_info -> (bonus, action_type, action_target), where bonus > 0
    res, bonus_info = [], []

    for p in list_of_points:
        # use to filter out social points at zen
        if params_ignore_zen and "zen" in p.tags and len(p.tags) == 1:
            continue

        p_tags_dict = dict.fromkeys(p.tags, p.ts)
        if (prev_ts is not None) and (p.ts - prev_ts > params_session_length or
                (params_split_by_tags and new_session_criterion(tags, p_tags_dict)) or
                (params_transition_new_session and p.transition is not None and p.transition.to not in tags)):
            res.append(create_session(session_ts, end_ts, color_ts, tags, bonus, bonus_all, penalized_tags, bonus_info))
            tags.clear()
            penalized_tags.clear()
            bonus_info = []
            session_ts, end_ts, color_ts, prev_ts, has_push = None, None, None, None, False
            bonus, bonus_all, penalty, last_l, last_r = 0, 0, 0, 0, 0

        # keep the first push only (used to remove duplicates if params_transition_bonus)
        if "push" in p.tags:
            if has_push:
                continue
            has_push = True

        prev_ts = p.ts

        # tags(tag) = maximal time for a point with this tag
        tags.update(p_tags_dict)
        tags[p.color] = p.ts

        # color_ts = time of first non-black point
        if (color_ts is None) and p.color != "black":
            color_ts = p.ts

        if p.ui == "touch" or p.ui == "desktop":
            tags[p.ui] = p.ts

        if session_ts is None:
            session_ts = p.ts

        end_ts = p.ts

        if p.transition is not None and p.color == "blue" and penalty == 0:
            penalty = params_st_penalty
            transition_from = getattr(p.transition, "from")
            # first social transition is penalized
            penalized_tags.add(transition_from)
            # punish the service as well for push and notifier
            if transition_from == "push" or transition_from == "notifier":
                penalized_tags.add(p.transition.to)
            bonus_info.append((-params_st_penalty, "penalty:" + p.action_type, p.action_target))
            # punish "all" if zen is not involved
            if "zen" not in p.tags:
                bonus_all -= penalty

        if p.transition is not None and not params_transition_bonus:
            continue

        if p.color == "red" or (p.color == "blue" and (p.ts - color_ts >= params_down_time or params_first_blue_bonus)):
            plen = color_interval[p.color] // 2
            cur_l = p.ts - plen
            cur_r = p.ts + plen
            # [cur_l, cur_r] - interval of p
            # [last_l, last_r] - previous session interval
            delta = 0
            if cur_l > last_r:
                last_l = cur_l
                last_r = cur_r
                delta = 2 * plen
            else:
                if cur_l < last_l:
                    delta = last_l - cur_l
                    last_l = cur_l
                if cur_r > last_r:
                    delta = cur_r - last_r
                    last_r = cur_r
            bonus += delta
            if delta > 0:
                bonus_info.append((delta, p.action_type, p.action_target))
                if "zen" not in p.tags:
                    bonus_all += delta

    if session_ts is not None:
        res.append(create_session(session_ts, end_ts, color_ts, tags, bonus, bonus_all, penalized_tags, bonus_info))

    return res


def create_session(session_ts, end_ts, color_ts, tags, clean_bonus, bonus_all, penalized_tags, bonus_info):
    res_tags = {"all": bonus_all}
    for tag, last_ts in tags.items():
        res_tags[tag] = (math.pow(params_black_transition_decay,
                                  color_ts - last_ts) if color_ts is not None and last_ts < color_ts else 1) * clean_bonus
    for tag in penalized_tags:
        res_tags[tag] -= params_st_penalty
    postprocess_tags(res_tags)
    return {"session_ts": session_ts, "session_end_ts": end_ts, "tags": res_tags, "bonus": clean_bonus, 'bonus_info': bonus_info}


def extend_tags(tags, tag, extra_tag):
    coeff = tags.get(tag)
    if coeff is None:
        return
    extra_coeff = tags.get(extra_tag)
    if extra_coeff is None or extra_coeff < coeff:
        tags[extra_tag] = coeff


def postprocess_tags(tags):
    extend_tags(tags, "mssngr_ios_browser", "browser_ios")
    extend_tags(tags, "mssngr_android_browser", "browser_android")
    extend_tags(tags, "mssngr_ios_searchapp", "pp_ios")
    extend_tags(tags, "mssngr_android_searchapp", "pp_android")
    extend_tags(tags, "iznanka_browser", "browser_android")
    extend_tags(tags, "oo", "oo+lk+epp+iznanka")
    extend_tags(tags, "lk", "oo+lk+epp+iznanka")
    extend_tags(tags, "epp", "oo+lk+epp+iznanka")
    extend_tags(tags, "iznanka_browser", "oo+lk+epp+iznanka")
    extend_tags(tags, "pp_android", "morda")
    extend_tags(tags, "pp_ios", "morda")
