use hahn;

declare $yt_pool as String;
declare $date as String;
declare $dst_path as String;

pragma yt.Pool = $yt_pool;
pragma yt.UseNativeYtTypes = "1";
pragma AnsiInForEmptyOrNullableItemsCollections;

$date_from = DateTime::Format("%Y-%m-%d")(cast($date as Date) - Interval("P6D"));
$date_shift = DateTime::Format("%Y-%m-%d")(cast($date as Date) - Interval("P4D"));
$date_to = $date;

$translate_answer_ids = ($yson) -> (
    Yson::ConvertTo($yson, List<Int64>?, Yson::Options(false as Strict, true as AutoConvert))
);

$moscow_date = ($eventtime) -> (
    cast(cast($eventtime + 3600*3 AS DateTime) AS Date)
);

$check_date = ($ts, $file_date) -> (
    if($moscow_date($ts) >= $file_date - Interval("P2D") and $moscow_date($ts) <= $file_date, $ts)
);

$choose_ts = ($client_ts_norm, $client_ts, $ts, $file_date) -> (
    $check_date($client_ts_norm / 1000, $file_date) ??
    $check_date($client_ts / 1000, $file_date) ??
    $check_date($ts / 1000, $file_date)
);

$EVENT_SHOW = "interview:show";
$EVENT_SUBMIT = "interview:submit";
$EVENT_CLOSE = "interview:close";

$src =
select
    TableName() as day,
    $choose_ts(client_ts_norm, client_ts, ts, cast(TableName() as Date)) as ts,
    strongest_id_prefix,
    strongest_id,
    yandexuid,
    event,
    item_id as question_id,
    if(
        origin_type is null,
        rid,
        rid || "/" || origin_type || ":" || (cast(origin_id as String) ?? "")
    ) as interview_id,
    $translate_answer_ids(_rest["answer_ids"]) as answer_ids,
    `uuid`,
    product,
    partner,
    integration,
from range(`//logs/zen-events-checked-log/1d`, $date_from, $date_to)
with columns Struct<_rest:Yson>
where
    event like "interview:%"
    and $choose_ts(client_ts_norm, client_ts, ts, cast(TableName() as Date)) is not null;

$shows =
select *
from (
    select min_by(TableRow(), question_timestamp)
    from (
        select
            s.*,
            ts as question_timestamp,
            answer_ids ?? ListCreate(Int64) as question_option_ids,
            `uuid` as question_uuid,
        from $src as s
        where event = $EVENT_SHOW
    )
    group by interview_id, question_id
)
flatten columns;

$submits =
select *
from (
    select max_by(TableRow(), answer_timestamp)
    from (
        select
            s.*,
            ts as answer_timestamp,
            if(event = $EVENT_SUBMIT, answer_ids ?? ListCreate(Int64), answer_ids) as answer_option_ids,
            `uuid` as answer_uuid,
        from $src as s
        where event in [$EVENT_SUBMIT, $EVENT_CLOSE]
    )
    group by interview_id, question_id
)
flatten columns;

$decode_screen_info = ($info) -> {
    $pr_map = ToDict(ListMap($info.answers, ($x) -> (($x.id, $x.positive_ratio))));
    $label_map = ToDict(ListMap($info.answers, ($x) -> (($x.id, $x.title))));
    return <|
        title: $info.title,
        ratio: if(ListLength(ListUniq(DictPayloads($pr_map))) > 1, $pr_map),
        labels: $label_map,
    |>;
};

$info =
select
    id as question_id,
    type as question_type,
    type = "multi_choice_text" as is_multiple_choice,
    $decode_screen_info(
        Yson::ConvertTo(
            gradation_choice_image_screen_info
            ?? gradation_stars_screen_info
            ?? message_screen_info
            ?? multi_choice_source_screen_info
            ?? multi_choice_text_screen_info
            ?? radio_choice_text_screen_info
            ?? single_choice_screen_info,
            Struct<
                title: String?,
                answers: List<
                    Struct<
                        id: Int64,
                        title: String?,
                        positive_ratio: Int64?
                    >
                >?
            >
        )
    ) as screen_info
from `//home/zen/dyntables/interviews_info` with inline;

$decode_ratio = ($answers, $map) -> (
    if($map is not null and ListLength($answers) = 1, cast($map[$answers[0]] as Integer))
);

$textify = ($list, $dict) -> (ListMap($list, ($x)->(Unwrap($dict[$x] ?? Just(null)))));

$result =
select
    q.day ?? a.day as day,
    q.interview_id ?? a.interview_id as interview_id,
    q.question_id ?? a.question_id as question_id,
    q.yandexuid ?? a.yandexuid as yandexuid,
    q.strongest_id_prefix ?? a.strongest_id_prefix as strongest_id_prefix,
    q.strongest_id ?? a.strongest_id as strongest_id,
    q.question_timestamp as question_timestamp,
    a.answer_timestamp as answer_timestamp,
    q.question_option_ids as question_option_ids,
    a.answer_option_ids as answer_option_ids,
    q.question_uuid ?? a.answer_uuid as `uuid`,
    q.product ?? a.product as product,
    q.partner ?? a.partner as partner,
    q.integration ?? a.integration as integration,
from $shows as q
full join $submits as a
on q.interview_id = a.interview_id and q.question_id = a.question_id;

insert into $dst_path with truncate
select unwrap(day) as day, t.*
without t.day
from $dst_path as t
where day < $date_shift
assume order by day, interview_id, question_timestamp, answer_timestamp;

insert into $dst_path
select
    unwrap(r.day) as day,
    r.*,
    question_type,
    screen_info.title as question_label,
    is_multiple_choice,
    $textify(question_option_ids, screen_info.labels) as question_option_labels,
    $textify(question_option_ids, screen_info.ratio) as question_option_values,
    $textify(answer_option_ids, screen_info.labels) as answer_option_labels,
    $decode_ratio(answer_option_ids, screen_info.ratio) as answer_value
without r.day
from $result as r
join $info as i
using (question_id)
where r.day >= $date_shift
order by day, interview_id, question_timestamp, answer_timestamp;
