import requests
import json
import codecs
import argparse


def argument_parser():
    parser = argparse.ArgumentParser(description='Get parameters')
    parser.add_argument('-i', dest='ticket_list', type=str, required=True, help='filepath to list of SbS tickets')
    parser.add_argument('-t', dest='oauth', type=str, required=True, help='oauth sbs token')
    parser.add_argument('-o', dest='output', type=str, required=True, help='output filepath')

    args = parser.parse_args()
    return args


def get_win_stat(query, sys_id):
    win_count = 0
    total_count = 0
    for comparison in query[u'comparisons']:
        if comparison[u'left'][u'sys_id'] == sys_id:
            win_count += comparison[u'left'][u'wins']
            total_count += comparison[u'left'][u'wins'] + comparison[u'right'][u'wins']
        if comparison[u'right'][u'sys_id'] == sys_id:
            win_count += comparison[u'right'][u'wins']
            total_count += comparison[u'left'][u'wins'] + comparison[u'right'][u'wins']
    return [str(win_count), str(total_count), str(1.0 * win_count/total_count)]


def get_row(ticket, query, sys_ids):
    row = [
        ticket.strip(),
        query[u'query'][u'text'],
        str(query[u'query'][u'region']),
        query[u'systems'][u'0'][u'html_url'],
        query[u'systems'][u'1'][u'html_url']
    ]

    total_exp_score = 0.0
    for sys_id in sys_ids:
        row += get_win_stat(query, sys_id)
        total_exp_score += query[u'systems'][sys_id][u'exp_score']

    for sys_id in sys_ids:
        row += [str(query[u'systems'][sys_id][u'exp_score']/total_exp_score)]

    return u'\t'.join(row)


def main():
    args = argument_parser()

    tickets = []
    with open(args.ticket_list, "r") as f:
        for line in f:
            tickets += [line]

    oauth = args.oauth

    sys_ids = [u'0', u'1']  # todo: fix this to arbitrary systems
    columns = [u'sbs_ticket', u'query', u'region', u'yandex_html', u'google_html']
    for sys_id in sys_ids:
        columns += [u'win-sys-{}'.format(sys_id), u'total-sys-{}'.format(sys_id), u'mean-win-sys-{}'.format(sys_id)]

    for sys_id in sys_ids:
        columns += [u'bt-sys-{}'.format(sys_id)]

    with codecs.open(args.output, 'w', 'utf-8') as f:
        f.write(u"\t".join(columns) + u"\n")
        for ticket in tickets:
            r = requests.get("https://sbs.yandex-team.ru/api/experiment/{}".format(ticket),
                             headers={"Content-Type": "application/json",
                                      "Authorization": "OAuth {}".format(oauth)},
                             verify=False
                             )
            data = json.loads(r.content)

            if u'results' in data:
                for row in data[u'results'][u'queries']:
                    result = get_row(ticket, row, sys_ids)
                    if result is not None:
                        f.write(result + "\n")


if __name__ == "__main__":
    main()

