#!/usr/bin/env python
# -*- coding: utf-8 -*-

import argparse
import pytz
import ast
import collections
from datetime import datetime
from itertools import repeat, izip
import multiprocessing
import urlparse
from geobase4 import Lookup

from nile.api.v1 import (
    filters as nf,
    aggregators as na,
    extractors as ne,
    clusters,
    Record,
)
import nile
import pandas as pd
import yt.wrapper as yt
import logging
logging.basicConfig(format='[%(asctime)s] %(filename)s[LINE:%(lineno)d] %(levelname)-8s %(message)s',
                    level=logging.ERROR)

JOB_NAME = 'HOME TRAIN TEST'

with open('b_list', 'r') as b_list:
    BLAND_BLOCKS = [x for x in b_list.read().split('\n')]

BLOCK_ACTIONS = {}
with open('block_actions', 'r') as actions_file:
    header = actions_file.readline().strip().split('\t')
    for line in actions_file:
        line_list = line.strip().split('\t')
        BLOCK_ACTIONS[line_list[0]] = {header[1+i]: int(line_list[1 + i]) for i in range(5)}


HITS_CATHEGORIES = ['hits_homework', 'hits_pets', 'hits_video', 'hits_ugcshop', 'hits_shop', 'hits_unmarked',
                    'hits_network', 'hits_teacher', 'hits_music', 'hits_books', 'hits_yellow_pages', 'hits_dating',
                    'hits_unknown', 'hits_porn', 'hits_weather', 'hits_auto', 'hits_news', 'hits_health', 'hits_laws',
                    'hits_bank', 'hits_games', 'hits_portal', 'hits_forum', 'hits_qa', 'hits_radio', 'hits_encyclopedia',
                    'hits_job', 'hits_carrier', 'hits_it', 'hits_mail', 'hits_woman', 'hits_post', 'hits_soft', 'hits_esoterics',
                    'hits_download', 'hits_gov', 'hits_tourism', 'hits_transport', 'hits_sport', 'hits_cooking', 'hits_tv',
                    'hits_education', 'hits_greet', 'hits_estate', 'hits_blogs', 'hits_building', 'hits_hard', 'hits_borda',
                    'hits_mother', 'hits_ads', 'hits_afisha', 'hits_dictionary', 'hits_eschool', 'hits_lottery', 'hits_pdd',
                    'hits_finance', 'hits_wallpapers', 'hits_chat', 'hits_maps', 'hits_image', 'hits_articles', 'hits_armor',
                    'hits_handmade', 'hits_farm', 'hits_travel', 'hits_religion', 'hits_ip', 'hits_speedtest', 'hits_localhost',
                    'hits_forex', 'hits_funny', 'hits_unpopular', 'hits_Ya_search_web', 'hits_Ya_search_touch', 'hits_Ya_search_app',
                    'hits_Ya_images', 'hits_Ya_video', 'hits_Ya_maps', 'hits_Ya_people', 'hits_Ya_news_web']


SEGMENTS_DICT = {
    'user_age_6s': [
        '0_17',
        '18_24',
        '25_34',
        '35_44',
        '45_54',
        '55_99',
    ],
    'probabilistic_segments': [
        '316', # 77%;    MLAL: money_cards
        '8',   # 76%;    (old) Маркетинг Firefox
        '315', # 72%;    MLAL: money_fines
        '380', # 54%;    MLAL: Маркетинг Радио модель 1
        '400', # 54%;    MLAL: yandex_radio_sites_list
        '460', # 52%;    MLAL: am_da535_speechkit_expand_interest_420
        '446', # 51%;    MLAL: speech-kit_sq
        '462', # 47%;    MLAL: am_da547_yamoney_gkh
        '101', # 47%;    Heavy TV viewers
        '317', # 47%;    MLAL: money_purse
        '318', # 46%;    MLAL: money_gamers
        '397', # 46%;    LAL: Пользователи планшетов Samsung OLD
        '314', # 45%;    MLAL: money_cassa
        '465', # 44%;    MLAL: am_da547_yamoney_fines
        '445', # 44%;    MLAL: speech-kit_competitors_sites
        '401', # 44%;    MLAL: radio_postclick
        '566', # 41%;    LAL: Есть дети
        '453', # 40%;    LAL: Посетители кинотеатров OLD
        '398', # 40%;    MLAL: yandex_radio_listeners
        '396', # 38%;    LAL: Пользователи смартфонов Samsung OLD
        '444', # 36%;    MLAL: speech-kit_partners_sites
        '466', # 36%;    MLAL: am_da547_yamoney_cassa
        '378', # 33%;    MLAL: maps_gismeteo
        '399', # 31%;    MLAL: background_music_listeners
        '408', # 31%;    LAL: Возраст 14+
        '461', # 28%;    MLAL: am_da535_speechkit_mobile_developers_p_a_r
        '430', # 26%;    MLAL: music_post_conversion
        '409', # 26%;    LAL: Возраст 14–40
        '467', # 26%;    LAL: Любители ТВ-передач про экстрасенсов
        '464', # 26%;    MLAL: am_da547_yamoney_gamers_console
        '126', # 26%;    LAL: Пользователи Apple (OLD)
        '581', # 24%;    LAL: Есть дети 3–6 лет
        '345', # 23%;    LAL: Владельцы котят (OLD)
        '123', # 22%;    LAL: Пользователи Android (OLD)
        '109', # 20%;    LAL: Молодожены
        '90',  # 19%;    LAL: Ведут домашнее хозяйство
        '188', # 19%;    LAL: E-shoppers (часто покупают в интернете)
        '102', # 19%;    Light TV viewers
        '463', # 18%;    MLAL: am_da547_yamoney_gamers_pc
        '434', # 17%;    LAL: Любители мелодрам
        '573', # 17%;    LAL: Любят русскую кухню
        '612', # 16%;    LAL: Планируют, ждут ребенка, есть дети до года
        '159', # 16%;    LAL: П�купатели автомобилей (семейные)
        '580', # 16%;    LAL: Есть дети 1–3 лет
        '429', # 15%;    MLAL: music_postclick
        '567', # 15%;    LAL: Есть дети до 1 года
        '377', # 15%;    MLAL: maps_google
        '564', # 14%;    LAL: Состоят в браке
        '576', # 14%;    LAL: Любят японскую кухню
        '92',  # 14%;    LAL: Женщины с детьми (OLD)
        '116', # 13%;    LAL: Путешествуют за границу
        '475', # 13%;    LAL: Планируют купить моб. телефон до 5 тыс. руб.
        '275', # 12%;    LAL: Владельцы планшетов
        '337', # 12%;    LAL: Женщины с детьми (0-6 лет) (OLD)
        '161', # 12%;    LAL: Ведут здоровый образ жизни
        '598', # 11%;    LAL: Работают
        '527', # 11%;    LAL: Сладкоежки (оптимизация Mondelez)
        '474', # 10%;    LAL: Планируют купить авто дороже 2 млн руб. OLD
        '571', # 10%;    LAL: Любят итальянскую кухню
        '306', # 10%;    LAL: Кофеманы
        '511', # 10%;    LAL: Любители фильмов ужасов OLD
        '199', # 9%;    LAL: Обеспеченные
        '112', # 9%;    LAL: Женщины с детьми (0-3 года) (OLD)
        '435', # 8%;    LAL: Любители триллеров
        '572', # 8%;    LAL: Любят кавказскую кухню
        '436', # 8%;    LAL: Любители фантастики
        '459', # 8%;    MLAL: Маркетинг_Советник
        '264', # 8%;    LAL: Тщательно следят за чистотой в доме
        '168', # 8%;    LAL: Выпендрежники (молодые женщины)
        '227', # 8%;    LAL: Много тратят на товары для детей
        '266', # 8%;    LAL: С доходом более 50 тыс. руб. в месяц
        '166', # 8%;    LAL: Активны в соцсетях
        '421', # 8%;    LAL: Геймеры OLD
        '577', # 8%;    LAL: Пользуются доставкой еды
        '263', # 8%;    LAL: Офисные работники (высокий доход, v2014b)
        '118', # 7%;    LAL: Офисные работники
        '422', # 7%;    LAL: Играют на мобильных устройствах OLD
        '174', # 7%;    LAL: Пользователи МТС (только)
        '579', # 7%;    LAL: Ухаживают за собой
        '427', # 7%;    LAL: Женщины с детьми (1-3 года) (OLD)
        '117', # 7%;    LAL: Менеджеры по закупкам
        '514', # 7%;    LAL: Театралы
        '260', # 7%;    LAL: Топ-менеджеры или руководители среднего звена
        '189', # 7%;    LAL:Женщины с детьми (до 1 года) или беременныеOLD
        '265', # 7%;    LAL: Любят готовить
        '578', # 7%;    LAL: Посещают рестораны
        '125', # 7%;    LAL: Пользователи iPad (OLD)
        '521', # 7%;    LAL: Планируют купить Ford Fiesta (оптимизация)
        '441', # 7%;    LAL: Интересуются брендом Renault
        '515', # 6%;    LAL: Киноманы
        '124', # 6%;    LAL: Пользователи iPhone (OLD)
        '113', # 6%;    LAL: Женщины с детьми (3-6 лет) (OLD)
        '186', # 6%;    LAL: Клабберы
        '424', # 6%;    LAL: Хотят купить автомобиль
        '122', # 6%;    LAL: Top-spenders (Ж) детских товаров (до 3) (OLD)
        '272', # 6%;    LAL: Модники
        '508', # 6%;    LAL: Посетители детских мероприятий
        '305', # 6%;    LAL: Пользователи планшетов Android (OLD)
        '319', # 6%;    MLAL: browser_anticrypta_150218
        '210', # 6%;    LAL: Ухаживают за собой (женщины) OLD
    ],
    'multiclass_segments': [
        ('114', '0'), # 37%;    Мультикласс по почте: нет данных
        ('114', '1'), # 37%;    Мультикласс по почте: alias
        ('114', '2'), # 37%;    Мультикласс по почте: owls
        ('114', '3'), # 37%;    Мультикласс по почте: bears
        ('116', '0'), # 24%;    Мультикласс по terminal.ru: нет данных
        ('116', '1'), # 24%;    Мультикласс по terminal.ru: техника для кухни
        ('116', '2'), # 24%;    Мультикласс по terminal.ru: аксессуары
        ('116', '3'), # 24%;    Мультикласс по terminal.ru: техника для дома
        ('116', '4'), # 24%;    Мультикласс по terminal.ru: телевизоры
        ('116', '5'), # 24%;    Мультикласс по terminal.ru: ноутбуки
        ('116', '6'), # 24%;    Мультикласс по terminal.ru: телефоны
        ('116', '7'), # 24%;    Мультикласс по terminal.ru: детские товары
    ],
    'interests_composite': [
        '17',  # 57%;    Интересы: Туризм
        '382', # 53%;    Интересы: Летние виды спорта
        '18',  # 52%;    Интересы: Кино
        '389', # 51%;    Интересы: Туры и экскурсии
        '70',  # 42%;    Интересы: Hi-Tech
        '14',  # 36%;    Интересы: B2B
        '68',  # 31%;    Интересы: Литература и Учебные материалы
        '146', # 31%;    Интересы: Hi Tech и Интернет
        '73',  # 27%;    Интересы: Заказной 5
        '13',  # 20%;    Интересы: Авто
        '23',  # 15%;    Интересы: Кулинария
        '16',  # 12%;    Интересы: Семья
        '56',  # 12%;    Интересы: Недвижимость
        '71',  # 11%;    Интересы: Спорт
        '139', # 10%;    Интересы: Бизнес/Реклама
        '190', # 9%;    Интересы: Авто (7 дней)
        '191', # 8%;    Интересы: Авто.Продажа автомобилей (7 дней)
        '15',  # 6%;    Интересы: Финансы
        '142', # 5%;    Интересы: Авто.Продажа автомобилей
        '12',  # 5%;    Интересы: Моб. связь и интернет
    ],
    'marketing_segments': [
        '96',  # 73%;    LAL: Keyword 217 = 315 (MLAL: money_fines)
        '22',  # 71%;    LAL: Модель по браузеру (со свопом) - АнтиКрипта
        '117', # 35%;
        '38',  # 26%;    ЯДеньги Касса
        '115', # 19%;    Интересы: Авто (217:13)
        '97',  # 19%;
        '163', # 19%;    Интересы: Авто
        '11',  # 18%;    adhoc: Пользователи смартфонов (для мобильного браузера), сегменты Рандом:0, Паспорт:1
        '194', # 17%;    Интересуются платной медициной
        '154', # 16%;    LAL: Автомобилисты
        '119', # 14%;
        '127', # 14%;
        '70',  # 14%;    Пользователи, интересующиеся новостями
        '73',  # 13%;    Пользователи, похожие на интерес 'Покупка' (9000103)
        '138', # 12%;    В Хамовниках: Москва
        '159', # 12%;    LAL: Женщины с детьми (0-6 лет)
        '124', # 12%;
        '139', # 11%;    Авиабилеты: похожие на пользователей авиабилетного Советника
        '123', # 11%;
        '116', # 11%;    Лучшие пользователи маркета
        '120', # 10%;
        '157', # 10%;    LAL: Кофеманы
        '150', # 10%;    Яндекс.Автобусы: люди посещавшие сайты, продающие билеты на автобусы онлайн
        '129', # 9%;
        '52',  # 9%;    Модель для категорий маркета. 6:Planshety
        '151', # 8%;    LAL по владельцам бизнеса для Яндекс.Справочника
        '162', # 8%;    LAL: Топ-спендеры (товары для детей)
        '49',  # 8%;    Модель для категорий маркета. 3:Detskaja_odezhda_i_obuv
        '1',   # 8%;    LAL: Модель по браузеру (со свопом)
        '132', # 7%;
        '112', # 7%;    Офисные работники (217:118)
        '51',  # 7%;    Модель для категорий маркета. 5:Zhenskaja_odezhda_i_obuv
        '103', # 7%;    LAL: Keyword 217 = 315
        '104', # 7%;    LAL: Keyword 217 = 315
        '128', # 7%;
        '83',  # 7%;    Тестовая модель смартфоны Android
        '102', # 6%;    LAL: Keyword 217 = 315
        '106', # 6%;    LAL: Keyword 217 = 315
        '130', # 6%;
        '113', # 6%;    Интересы: Финансы (217:15)
        '84',  # 6%;    Тестовая модель смартфоны iPhone
        '105', # 6%;    LAL: Keyword 217 = 315
        '114', # 6%;    Уход за собой (женщины) (217:210)
        '50',  # 6%;    Модель для категорий маркета. 4:Telefony
        '98',  # 5%;
        '152', # 5%;    Авиабилеты: пользователи, похожие на тех, у кого были с высокой точностью авиабилеты за 3 года
        '122', # 5%;
        '111', # 5%;    Студенты (217:88)
        '155', # 5%;    LAL: Дачники
        '101', # 5%;    LAL: Keyword 217 = 315
            ],
    'interests_longterm': [
        '9003579', # 59%;    Отдых
        '9000088', # 57%;    Отдых/Туризм
        '9000006', # 56%;    Бизнес
        '9000011', # 55%;    СМИ
        '9000003', # 53%;    Общество
        '9000090', # 53%;    Спорт
        '9000001', # 51%;    Культура
        '9000399', # 51%;    Отдых/Туризм/Туры и экскурсии
        '9000117', # 48%;    СМИ/Телевидение
        '9000009', # 45%;    Дом
        '9000037', # 44%;    Общество/Власть
        '9000118', # 42%;    СМИ/Периодика
        '9000004', # 42%;    Hi-Tech
        '9003349', # 40%;    Справки
        '9000020', # 37%;    Развлечения/Кино
        '9000041', # 36%;    Общество/Страны и города
        '9000007', # 36%;    Бизнес/Производство и поставки
        '9000038', # 36%;    Общество/Власть/Силовые структуры
        '9000069', # 32%;    Бизнес/Финансы
        '9000005', # 31%;    Hi-Tech/Интернет
        '9000018', # 30%;    Развлечения/Литература
        '9000008', # 30%;    Развлечения
        '9000002', # 29%;    Учеба
        '9002991', # 25%;    Справки/Погода
        '9000096', # 22%;    Дом/Здоровье
        '9000019', # 21%;    Развлечения/Музыка
        '9002877', # 21%;    Отдых/Где развлечься
        '9000099', # 20%;    Авто
        '9000072', # 19%;    Бизнес/Деловые услуги
        '9000257', # 19%;    Общество/Страны и города/Краеведение
        '9000082', # 18%;    Бизнес/Производство и поставки/Агропром
        '9002866', # 18%;    Дом/Семья
        '9000066', # 16%;    Справки/Транспорт
        '9000091', # 16%;    Развлечения/Личная жизнь
        '9000039', # 16%;    Общество/НКО
        '9000095', # 15%;    Дом/Кулинария
        '9003281', # 15%;    Учеба/Науки
        '9000333', # 14%;    Бизнес/Строительство
        '9000103', # 14%;    Дом/Покупки
        '9000105', # 14%;    Дом/Мода и красота
        '9000146', # 13%;    Развлечения/Юмор
        '9000068', # 12%;    Бизнес/Недвижимость
        '9002873', # 11%;    Дом/Семья/Родителям
        '9000012', # 10%;    Развлечения/Непознанное
        '9000073', # 10%;    Бизнес/Реклама
        '9000668', # 9%;    Дом/Все для праздника
        '9000635', # 9%;    Дом/Здоровье/Специализированная медицинская п
        '9000437', # 8%;    Дом/Кулинария/Рецепты
        '9000489', # 8%;    СМИ/Телевидение/Телестудии и телекомпании
        '9000116', # 8%;    СМИ/Радио
        '9000098', # 8%;    Отдых/Хобби
        '9000043', # 7%;    Общество/Религия
        '9000600', # 7%;    Учеба/Науки/Гуманитарные науки
        '9000165', # 7%;    Развлечения/Музыка/Поп-музыка
        '9000017', # 7%;    Культура/Театры
        '9003186', # 7%;    СМИ/Телевидение/Программы ТВ
        '9000245', # 7%;    Общество/НКО/Защита гражданских прав
        '9000026', # 6%;    Учеба/Среднее образование
        '9000461', # 5%;    Авто/Продажа автомобилей
        '9000422', # 5%;    Развлечения/Личная жизнь/Эротика и секс
        '9000485', # 5%;    СМИ/Радио/Радиостанции
        '9000311', # 5%;    Hi-Tech/Мобильная связь
        '9000087', # 5%;    Развлечения/Игры
        '9000337', # 5%;    Бизнес/Финансы/Банки
        '9000411', # 5%;    Спорт/Футбол
    ],
    'heuristic_segments': [
        '595', # 43%;    Проводят много времени в интернете
        '547', # 33%;    Поколение X
        '548', # 28%;    Поколение Миллениум
        '36',  # 27%;    Внутренние путешественники
        '522', # 18%;    Активные пользователи мобильного интернета
        '519', # 17%;    Автомобилисты
        '600', # 16%;    Ходят в кино
        '596', # 9%;    Проводят мало времени в интернете
        '603', # 6%;    Ищут работу
        '541', # 5%;    Дачники
        '606', # 5%;    Используют электронный кошелек
        '613', # 5%;    Играют на мобильных устройствах
        '37',  # 4%;    Международные путешественники
    ]
}
BLOCK_FEATURES = {
    'clicked': 0,
    'clicked_freq': 0,
    'clicked_top': 0,
    'count': 0,
    'ctr': 0,
    'ctr_top': 0,
    'showed_freq': 0,
    'total_clicks': 0,
    'block_freq_days': 0,
    'block_long_love': 0,
    'block_love': 0,
    'block_unique_days': 0,
    'block_weight_love': 0,
    'love_top': 0,
    'weekend_freq': 0,
}
EMPTY_CONST = -100
ZERO_CONST = 0


def show_day_part(ts, pytz_timezone):
    hour = datetime.fromtimestamp(ts, pytz_timezone).hour
    if hour > 21 or hour <= 3:
        return 'night'
    elif hour > 17:
        return 'evening'
    elif hour > 12:
        return 'afternoon'
    elif hour > 7:
        return 'morning'
    else:
        return 'early_morning'


def e_shoppers(heuristic_segments):
    if heuristic_segments:
        score = 0
        if '529' in heuristic_segments:
            score += heuristic_segments['529']
        if '530' in heuristic_segments:
            score += heuristic_segments['530']
        if '531' in heuristic_segments:
            score += heuristic_segments['531']
        if '532' in heuristic_segments:
            score += heuristic_segments['532']
        if '533' in heuristic_segments:
            score += heuristic_segments['533']
        if '534' in heuristic_segments:
            score += heuristic_segments['534']
        if '558' in heuristic_segments:
            score += 5 * heuristic_segments['558']
        return score
    else:
        return 0


def age_score(age_dict):
    if age_dict:
        score = age_dict['0_17']
        score += 2 * age_dict['18_24']
        score += 3 * age_dict['25_34']
        score += 4 * age_dict['35_44']
        score += 5 * age_dict['45_54']
        score += 6 * age_dict['55_99']
        return score
    else:
        return EMPTY_CONST


def income_score(income_dict):
    if income_dict:
        score = income_dict['A']
        score += 2 * income_dict['B']
        score += 3 * income_dict['C']
        return score
    else:
        return EMPTY_CONST


def find_segm(s_id, s_dict):
    if s_dict:
        if s_id in s_dict:
            return s_dict[s_id]
        else:
            return ZERO_CONST
    else:
        return EMPTY_CONST


def find_probasegm(s_id, s_dict):
    if s_dict:
        if s_id in s_dict:
            if '1' in s_dict[s_id]:
                return s_dict[s_id]['1']
            elif '0' in s_dict[s_id]:
                return 1 - s_dict[s_id]['0']
            else:
                return ZERO_CONST
        else:
            return ZERO_CONST
    else:
        return EMPTY_CONST


def find_multisegm(s_id1, s_id2, s_dict):
    if s_dict:
        if s_id1 in s_dict:
            if s_id2 in s_dict[s_id1]:
                return s_dict[s_id1][s_id2]
            else:
                return ZERO_CONST
        else:
            return ZERO_CONST
    else:
        return EMPTY_CONST


def prepare_user_age(user_age_6s):
    if user_age_6s:
        user_age_6s['45_54'] += user_age_6s['55_99']
        user_age_6s['55_99'] = 0
        return user_age_6s
    else:
        return None


def click_type(dwell):
    if dwell >= 30:
        return 'long'
    elif dwell < 15:
        return 'short'
    else:
        return 'normal'


def weight_target(block_name, watch_time_sum, all_clicks):
    clicks_dict = {
        'long': 0,
        'normal': 0,
        'short': 0,
        'dynamic': 0,
        'longwatch': int(watch_time_sum >= 8000)
    }
    for click in all_clicks:
        if click['dynamic'] == 1:
            clicks_dict['dynamic'] += 1
        elif (click['url'] is None) or (len(click['url']) == 0):
            if (block_name == 'tv') and ('links' in click['path']):
                clicks_dict[click_type(click['dwell'])] += 1
            elif block_name == 'tv_online':
                clicks_dict[click_type(click['dwell'])] += 1
            elif block_name == 'poi':
                clicks_dict[click_type(click['dwell'])] += 1
            else:
                clicks_dict['dynamic'] += 1
        else:
            clicks_dict[click_type(click['dwell'])] += 1
    max_weight = -100
    for key in clicks_dict:
        if clicks_dict[key] > 0:
            if BLOCK_ACTIONS[block_name][key] > max_weight:
                max_weight = BLOCK_ACTIONS[block_name][key]
    if max_weight == -100:
        max_weight = 0
    return max_weight, clicks_dict


def make_features(
    ts,
    timezone,
    block_path,
    gender,
    yandex_loyalty,
    user_age_6s,
    income_segments,
    heuristic_segments,
    multiclass_segments,
    probabilistic_segments,
    interests_composite,
    marketing_segments,
    interests_longterm,
    mean_clicked,
    mean_realshow_blocks,
    req_clicked_freq,
    req_count,
    blocks_dict,
    mean_blocks,
    hits,
):
    features_dict = {}
    user_age_6s = prepare_user_age(user_age_6s)
    try:
        pytz_timezone = pytz.timezone(timezone)
    except:
        pytz_timezone = pytz.timezone('Europe/Moscow')
    features_dict['week_day'] = str(datetime.fromtimestamp(ts, pytz_timezone).isoweekday())
    features_dict['weekend'] = int(features_dict['week_day'] in ['6', '7'])
    features_dict['day_time'] = show_day_part(ts, pytz_timezone)
    if mean_clicked is not None:
        features_dict['personal__mean_clicked'] = mean_clicked
    else:
        features_dict['personal__mean_clicked'] = EMPTY_CONST
    if mean_realshow_blocks is not None:
        features_dict['personal__mean_realshow_blocks'] = mean_realshow_blocks
    else:
        features_dict['personal__mean_realshow_blocks'] = EMPTY_CONST
    if req_clicked_freq is not None:
        features_dict['personal__req_clicked_freq'] = req_clicked_freq
    else:
        features_dict['personal__req_clicked_freq'] = EMPTY_CONST
    if req_count is not None:
        features_dict['personal__req_count'] = req_count
    else:
        features_dict['personal__req_count'] = EMPTY_CONST
    if mean_blocks is not None:
        features_dict['personal__mean_blocks'] = mean_blocks
    else:
        features_dict['personal__mean_blocks'] = EMPTY_CONST
    if gender:
        features_dict['man_gender'] = gender['m']
    else:
        features_dict['man_gender'] = 0.5
    features_dict['yandex_loyalty'] = yandex_loyalty
    features_dict['income_segment_score'] = income_score(income_segments)
    features_dict['age_score'] = age_score(user_age_6s)
    features_dict['e_shoppers'] = e_shoppers(heuristic_segments)

    if hits:
        for h in hits:
            label = "hits_" + h.replace(" ", "_")
            features_dict[label] = hits[h]
    for h in HITS_CATHEGORIES:
        if not h in features_dict:
            features_dict[h] = EMPTY_CONST

    if blocks_dict is not None:
        if block_path in blocks_dict:
            for block_feat in BLOCK_FEATURES:
                features_dict['block__{}'.format(block_feat)] = blocks_dict[block_path][block_feat]
        else:
            for block_feat in BLOCK_FEATURES:
                features_dict['block__{}'.format(block_feat)] = BLOCK_FEATURES[block_feat]
    else:
        for block_feat in BLOCK_FEATURES:
            features_dict['block__{}'.format(block_feat)] = EMPTY_CONST
    for s_id in SEGMENTS_DICT['user_age_6s']:
        if user_age_6s:
            features_dict['user_age_{}'.format(s_id)] = user_age_6s[s_id]
        else:
            features_dict['user_age_{}'.format(s_id)] = None

    for s_id in SEGMENTS_DICT['heuristic_segments']:
        features_dict['heuristic_segments_{}'.format(s_id)] = find_segm(s_id, heuristic_segments)

    for s_id in SEGMENTS_DICT['marketing_segments']:
        features_dict['marketing_segments_{}'.format(s_id)] = find_segm(s_id, marketing_segments)

    for s_id in SEGMENTS_DICT['interests_longterm']:
        features_dict['interests_longterm_{}'.format(s_id)] = find_segm(s_id, interests_longterm)

    for s_id1, s_id2 in SEGMENTS_DICT['multiclass_segments']:
        features_dict['multiclass_segments_{}_{}'.format(s_id1, s_id2)] = find_multisegm(s_id1, s_id2, multiclass_segments)

    for s_id in SEGMENTS_DICT['probabilistic_segments']:
        features_dict['probabilistic_segments_{}'.format(s_id)] = find_probasegm(s_id, probabilistic_segments)

    for s_id in SEGMENTS_DICT['interests_composite']:
        features_dict['interests_composite_{}'.format(s_id)] = find_probasegm(s_id, interests_composite)
    return features_dict


def geobase_reduce(groups):
    gb = Lookup('geodata4.bin')
    for key, recs in groups:
        region = key.region
        date = key.date
        try:
            timezone = gb.region_by_id(region).timezone
            test = pytz.timezone(timezone)
        except:
            timezone = 'error'
        if timezone == '':
            timezone = 'error'
        for request in recs:
            reqid = request['reqid']
            clicks_in_req = request['clicks_in_req']
            yield Record(
                reqid=reqid,
                region=region,
                date=date,
                timezone=timezone,
                clicks_in_req=clicks_in_req,
            )


def argument_parser():
    parser = argparse.ArgumentParser(description='Get parameters')
    parser.add_argument(
        '-p',
        dest='pool',
        type=str,
        help='you hahn pool',
    )
    parser.add_argument(
        '-lr',
        dest='lpath',
        type=str,
        help='logs path',
    )
    parser.add_argument(
        '-r',
        dest='rpath',
        type=str,
        help='result path',
    )
    parser.add_argument(
        '-d1',
        dest='date1',
        type=str,
        help='date1 format yyyy-mm-dd',
    )
    parser.add_argument(
        '-d2',
        dest='date2',
        type=str,
        help='date2 format yyyy-mm-dd',
    )
    parser.add_argument(
        '-control',
        dest='control',
        type=bool,
        default=False,
        required=False,
        help='If true, picks only control users',
    )
    args = parser.parse_args()
    return args


def main():
    args = argument_parser()
    date1 = args.date1
    date2 = args.date2
    result_path = args.rpath
    log_root = args.lpath
    cluster = clusters.Hahn(pool=args.pool).env(templates=dict(
            result_path=result_path,
            log_root=log_root,
            tmp_files='//home/search-functionality/syudin/tmp',
        )
    )
    dates = [i.strftime('%Y-%m-%d') for i in pd.date_range(date1, date2)]

    yt.config.set_proxy('hahn')
    external_files = [
        nile.files.StatboxDict('geodata4.bin'),
    ]
    dates_done = list(set(dates).intersection(set(yt.list(log_root))))
    dates_str = '{' + ','.join([x for x in dates_done]) + '}'
    job = cluster.job(JOB_NAME + ': result').env(
        templates=dict(
            dates=dates_str,
        )
    )
    log = job.table('$log_root/@dates')
    if args.control:
        filtered_log = log.filter(
            nf.and_(
                nf.custom(lambda x: x < 1000, 'position'),
                nf.custom(lambda x: '54863' in x, 'enabled_test_ids'),
                nf.custom(lambda x: x in BLAND_BLOCKS, 'path'),
            )
        ).project(
            ne.all(),
            true_num_clicks=ne.custom(lambda x: len(x), 'clicks')
        )
    else:
        filtered_log = log.filter(
            nf.and_(
                nf.custom(lambda x: x < 1000, 'position'),
                nf.custom(lambda x: x in BLAND_BLOCKS, 'path'),
            )
        ).project(
            ne.all(),
            true_num_clicks=ne.custom(lambda x: len(x), 'clicks')
        )

    reqid_aggr = filtered_log.groupby(
        'reqid',
        'region',
        'date',
    ).aggregate(
        clicks_in_req=na.sum('true_num_clicks')
    ).groupby(
        'region',
        'date',
    ).reduce(
        geobase_reduce,
        files=external_files,
    )

    result = filtered_log.join(
        reqid_aggr,
        by=('reqid', 'date', 'region'),
        type='left'
    ).filter(
        nf.and_(
            nf.custom(lambda x: x > 0, 'clicks_in_req'),
            nf.custom(lambda x: x != 'error', 'timezone'),
        )
    ).project(
        reqid='reqid',
        position='position',
        subkey='date',
        true_num_clicks='true_num_clicks',
        realshow='realshow',
        uid='uid',
        path='path',
        region='region',
        parent='parent',
        timezone='timezone',
        label_clicked=ne.custom(lambda c: int(c > 0), 'true_num_clicks'),
        features_dict=ne.custom(
            make_features,
            'ts',
            'timezone',
            'path',
            'gender',
            'yandex_loyalty',
            'user_age_6s',
            'income_segments',
            'heuristic_segments',
            'multiclass_segments',
            'probabilistic_segments',
            'interests_composite',
            'marketing_segments',
            'interests_longterm',
            'mean_clicked',
            'mean_realshow_blocks',
            'req_clicked_freq',
            'req_count',
            'blocks_dict',
            'mean_blocks',
            'hits',
        ),
        label_weightclicked=ne.custom(lambda p, w, c: weight_target(p, w, c)[0], 'path', 'watch_time_sum', 'clicks'),
        label_longclicked=ne.custom(lambda p, w, c: int(weight_target(p, w, c)[1]['long'] > 0), 'path', 'watch_time_sum', 'clicks'),
    ).put('$result_path')
    job.run()

if __name__ == '__main__':
    main()
