from itertools import chain, combinations

from nile.api.v1 import (
    Record,
    clusters,
    statface,
    extractors as ne,
    filters as nf,
    aggregators as na,
    modified_schema,
    with_hints,
    cli
)

from qb2.api.v1 import (
    extractors as qe,
    resources as qr,
    typing as qt
)


class AggregationType:
    def __init__(self, name, filters, is_distinct, predicate=None):
        self.filters = filters
        self.is_distinct = is_distinct
        self.name = name
        self.predicate = predicate

    def get_filters(self):
        return nf.or_(*[nf.equals("EventName", x) for x in self.filters])

    def event_predicate(self):
        name_filter = self.get_filters()
        if self.predicate:
            return nf.and_(name_filter, self.predicate)
        return name_filter

    def get_count_function(self):
        if self.is_distinct:
            return na.count_distinct("DeviceID", predicate=self.event_predicate())
        else:
            return na.count(predicate=self.event_predicate())

    def get_any_function(self):
        return na.any(self.name)

    def get_sum_function(self):
        return na.sum(self.name)


def parse_event_value(raw_dict):
    try:
        event_dict = eval(raw_dict.replace("false", "False").replace("true", "True").replace("null", "'none'"))
        return event_dict
    except:
        return {}


def check_param_equal(event_value, param, target):
    return parse_event_value(event_value).get(param, "none") in target


def powerset(iterable):
    "powerset([1,2,3]) --> () (1,) (2,) (3,) (1,2) (1,3) (2,3) (1,2,3)"
    s = list(iterable)
    return chain.from_iterable(combinations(s, r) for r in range(len(s)+1))


def get_android_version(text_data):
    if text_data is None:
        return "other"
    vals = text_data.split(".")
    android_versions = ["4.0", "4.1", "4.2", "4.3", "4.4", "5.0", "5.1", "6.0", "7.0", "7.1", "8.0", "8.1", "9.0", "10.0", "11.0"]

    normalized_version = ".".join(vals[:2])
    if normalized_version == "9":
        normalized_version = "9.0"
    if normalized_version == "Q" or normalized_version == "10":
        normalized_version = "10.0"
    if normalized_version == "11":
        normalized_version = "11.0"

    if normalized_version in android_versions:
        return normalized_version
    else:
        return "other"


@cli.statinfra_job
def make_job(job, options, statface_client):
    report = statface.report.StatfaceReport(
        path="Distribution/Adhoc/searchlib/funnel_test",
        scale="daily",
        replace_mask="fielddate",
        client=statface_client
    )

    squeeze_table = job.table(
        "//home/searchlib/squeeze/@dates"
    ).filter(
        nf.equals("EventType", 4),  # EVENT_CLIENT
        nf.custom(lambda x: options.dates[0] <= x <= options.dates[-1], "EventDate")
    ).project(
        "DeviceID", "EventName", "EventValue",
        qe.geo_region(
            'country_id',
            qr.geo.subregions_by_type(region_type=qr.geo.RegionTypes.COUNTRY),
            disjoint=True,
            geo_id="RegionID"
        ),
        apikey=ne.custom(lambda x: str(x), "APIKey").with_type(str),
        fielddate="EventDate",
        version=ne.custom(lambda x: parse_event_value(x).get("version", "none"), "EventValue").with_type(str),
        android_version=ne.custom(get_android_version, "OSVersion")
    ).project(
        ne.all(),
        country_id=ne.custom(lambda x: x if x is not None else -1, "country_id").with_type(int)
    )

    aggregation_types = [
        AggregationType("searchlib_widget_dayuse", ["searchlib_widget_dayuse"], True),
        AggregationType("searchlib_bar_dayuse", ["searchlib_dayuse"], True),
        AggregationType("searchlib_bar_visible", ["searchlib_dayuse"], True,
                        nf.custom(lambda x: check_param_equal(x, "bar_hidden", {False}), "EventValue")),
        AggregationType("searchlib_dayuse", ["searchlib_widget_dayuse", "searchlib_dayuse"], True),
        AggregationType("searchlib_bar_clicked", ["searchlib_bar_clicked", "searchlib_surface_clicked"], True,
                        nf.custom(lambda x, y: y == 'searchlib_surface_clicked' and check_param_equal(x, "kind", {"bar", "none"})
                                               or y == 'searchlib_bar_clicked', "EventValue", "EventName")),
        AggregationType("searchlib_widget_clicked", ["searchlib_widget_clicked", "searchlib_surface_clicked"], True,
                        nf.custom(lambda x, y: y == 'searchlib_surface_clicked' and check_param_equal(x, "kind", {"widget"})
                                                or y == 'searchlib_widget_clicked', "EventValue", "EventName")),
        AggregationType("searchlib_bar_clicks", ["searchlib_bar_clicked", "searchlib_surface_clicked"], False,
                        nf.custom(lambda x, y: y == 'searchlib_surface_clicked' and check_param_equal(x, "kind", {"bar", "none"})
                                               or y == 'searchlib_bar_clicked', "EventValue", "EventName")),
        AggregationType("searchlib_widget_clicks", ["searchlib_widget_clicked", "searchlib_surface_clicked"], False,
                        nf.custom(lambda x, y: y == 'searchlib_surface_clicked' and check_param_equal(x, "kind", {"widget"})
                                               or y == 'searchlib_widget_clicked', "EventValue", "EventName")),
        AggregationType("searchlib_splash_shown", ["searchlib_splash_shown"], False),
        AggregationType("searchlib_bar_splash_shown", ["searchlib_splash_shown"], False,
                        nf.custom(lambda x: check_param_equal(x, "kind", {"bar", "barwidget"}), "EventValue")),
        AggregationType("searchlib_widget_splash_shown", ["searchlib_splash_shown"], False,
                        nf.custom(lambda x: check_param_equal(x, "kind", {"widget", "barwidget"}), "EventValue")),
        AggregationType("searchlib_splash_yes", ["searchlib_splash_action"], False,
                        nf.custom(lambda x: check_param_equal(x, "action", {"ok", "yes"}), "EventValue")),
        AggregationType("searchlib_bar_splash_yes", ["searchlib_splash_action"], False,
                        nf.custom(lambda x: (check_param_equal(x, "kind", {"bar", "barwidget"})
                                             and check_param_equal(x, "action", {"ok", "yes"})), "EventValue")),
        AggregationType("searchlib_widget_splash_yes", ["searchlib_splash_action"], False,
                        nf.custom(lambda x: (check_param_equal(x, "kind", {"widget", "barwidget"})
                                             and check_param_equal(x, "action", {"ok", "yes"})), "EventValue")),
        AggregationType("searchlib_enable_bar", ["searchlib_enable_bar"], False,
                        nf.custom(lambda x: check_param_equal(x, "enable", {True}), "EventValue")),
        AggregationType("searchlib_enable_widget", ["searchlib_enable_widget", "searchlib_widget_enable"], False,
                        nf.custom(lambda x: check_param_equal(x, "enable", {True}), "EventValue")),
        AggregationType("searchlib_bar_search_clicks", ["searchlib_search_clicked"], False,
                        nf.custom(lambda x: check_param_equal(x, "kind", {"bar", "none"}), "EventValue")),
        AggregationType("searchlib_widget_search_clicks", ["searchlib_search_clicked"], False,
                        nf.custom(lambda x: check_param_equal(x, "kind", {"widget"}), "EventValue"))
    ]

    not_pp_or_bro = squeeze_table.filter(
            nf.custom(lambda x: x not in {"10321", "106400", "106450", "372509", "149390"}, "apikey")
    )

    calculations = []
    all_dimensions = ["country_id", "version", "apikey", "android_version"]
    for cur_set in powerset(all_dimensions):
        project_dict = {dim: ne.const("_total_") for dim in all_dimensions if dim not in cur_set}
        calculations += [
            squeeze_table.groupby(
                "fielddate", *cur_set
            ).aggregate(
                **{event_type.name: event_type.get_count_function() for event_type in aggregation_types}
            ).project(
                ne.all(),
                **project_dict
            )
        ]

    all_dimensions = ["country_id", "version", "android_version"]
    for cur_set in powerset(all_dimensions):
        project_dict = {dim: ne.const("_total_") for dim in all_dimensions if dim not in cur_set}
        calculations += [
            not_pp_or_bro.groupby(
                "fielddate", *cur_set
            ).aggregate(
                **{event_type.name: event_type.get_count_function() for event_type in aggregation_types}
            ).project(
                ne.all(),
                apikey=ne.const("not (PP or Browser)"),
                **project_dict
            )
        ]

    job.concat(
        *calculations
    ).put(
        "//home/searchlib/funnel/{}".format(options.dates[-1])
    ).publish(
        report, remote_mode=True
    )

    return job


if __name__ == "__main__":
    cli.run()
