# coding=utf-8
import logging

from yql.api.v1.client import YqlClient
from yql.client.results import CantGetResultsException

query = """
$tables = [
"//logs/appmetrica-external-events/1d/{0}",
"//logs/appmetrica-yandex-events/1d/{0}",
"//logs/superapp-metrika-mobile-log/1d/{0}",
"//logs/browser-metrika-mobile-log/1d/{0}",
"//logs/disk-metrika-mobile-log/1d/{0}",
"//logs/navi-metrika-mobile-log/1d/{0}",
];

$apikeys = AsList({1});

$int_apikeys = ListMap($apikeys, ($x) -> {{ return cast($x as UInt32); }});

$date_format = DateTime::Format("%Y-%m-%d");

INSERT INTO `//home/searchlib/squeeze/{0}` WITH TRUNCATE
SELECT APIKey, AppID, AppVersionName,
      StartTimestamp, `UUID`, DeviceID, Manufacturer,
      Model, OSVersion, SendTimestamp, ReceiveTimestamp,
      SessionID, EventTimestamp, EventType, EventName,
      EventValue, $date_format(DateTime::FromSeconds(cast((EventDate + EventTimeZone) as UInt32))) as EventDate,
      EventTimeZone, CAST(RegionID as Int64) as RegionID
FROM EACH($tables)
WHERE EventName LIKE 'searchlib%'
AND APIKey IN
ToSet(ListCollect($int_apikeys))
AND EventName != 'searchlib_trend_changed'
AND EventName != 'searchlib_surface_received'
AND EventName != "searchlib_informers_update_started"
AND EventName != "searchlib_informers_updated";
COMMIT;
"""


def argument_parser():
    import argparse
    parser = argparse.ArgumentParser(description='Get parameters')
    parser.add_argument(
        '-d',
        dest='date',
        help='date of table',
    )

    parser.add_argument(
        '--apikeys',
        dest='apikeys',
        help='path to file with apikeys'
    )

    args = parser.parse_args()
    return args


def prepare_apikeys(apikeys_filename):
    apikeys = []
    with open(apikeys_filename, "r") as f:
        for line in f:
            apikeys += ['\'{}\''.format(line.strip())]
    return ','.join(apikeys)


class YTException(Exception):
    """ Ошибка при выполнении yt-операций. """
    pass


if __name__ == "__main__":
    args = argument_parser()

    query = query.format(args.date, prepare_apikeys(args.apikeys))
    print(query)

    with YqlClient(db='arnold') as yql_client:
        query = yql_client.query(query, syntax_version=1)
        query.run()
        print ("YQL link: {0}".format(query.web_url))
        print ("YQL shared link: {0}".format(query.share_url))
        result = query.get_results()

        try:
            request_status = query.status
        except CantGetResultsException:
            logging.error('Perhaps yql query was interrupted')
            raise YTException('Perhaps yql query was interrupted')

        if request_status == "ABORTED":
            logging.error('YQL query was interrupted')
            raise YTException('YQL query was interrupted')

        if request_status.endswith('ERROR'):
            logging.error('Error while executing YQL query')
            for error in query.errors:
                for message in str(error).splitlines():
                    logging.error(message.strip())
            raise YTException('Error in YQL api request: %s' % query.errors[0].message)

        logging.info("YQL query complete")
