from nile.api.v1 import (
    Record,
    clusters,
    statface,
    extractors as ne,
    filters as nf,
    aggregators as na,
    modified_schema ,
    with_hints,
    cli
)


def get_value_dec(key, default_value):
    def get_value(raw_dict):
        event_dict = eval(raw_dict.replace("false", "False").replace("true", "True"))
        return event_dict.get(key, default_value)
    return get_value


def get_mappers(table_date):
    @with_hints(output_schema=dict(fielddate=str, splash_opt_in_shown_count=int, splash_opt_out_shown_count=int))
    def get_splash_shown_count(records):
        d = {"fielddate": table_date}
        for record in records:
            if record["opt_in"]:
                d["splash_opt_in_shown_count"] = record["count"]
            else:
                d["splash_opt_out_shown_count"] = record["count"]
        yield Record.from_dict(d)

    @with_hints(output_schema=dict(fielddate=str, splash_action_no=int, splash_action_install=int))
    def get_action_no(records):
        d = {"fielddate": table_date, "splash_action_no": 0, "splash_action_install": 0}
        for record in records:
            if record["opt_in"] and (record["splash_action_type"] == "back" or record["splash_action_type"] == "no"):
                d["splash_action_no"] += record["count"]
            if not record["opt_in"] or record["splash_action_type"] == "yes":
                d["splash_action_install"] += record["count"]
        yield Record.from_dict(d)

    return get_splash_shown_count, get_action_no


def get_table_name(timestamp, day_delta):
    import datetime
    import time
    need_date = (datetime.datetime.fromtimestamp(timestamp) - datetime.timedelta(day_delta)).date()
    need_date_mk = int(time.mktime(need_date.timetuple()))
    return need_date_mk, need_date.strftime('%Y-%m-%d')


@cli.statinfra_job
def make_job(job, options, statface_client):
    table_date = options.dates[0]
    get_splash_shown_count, get_action_no = get_mappers(table_date)

    report = statface.report.StatfaceReport(path="Distribution/Adhoc/SL_annoyance",
                                            scale="daily",
                                            replace_mask=("fielddate"),
                                            client=statface_client)

    squeeze_table = job.table("//home/searchlib/squeeze/$date")

    gb_splash_shown = squeeze_table\
        .filter(
            nf.equals("EventName", "searchlib_splash_shown")
        ).project(
            opt_in=ne.custom(get_value_dec("opt_in", "null"), "EventValue").add_hints(type=bool)
        ).groupby("opt_in")\
        .aggregate(count=na.count())\
        .map(get_splash_shown_count)

    splash_action_table = squeeze_table\
        .filter(
            nf.equals("EventName", "searchlib_splash_action")
        ).project(
            "UUID",
            "EventTimestamp",
            splash_action_type=ne.custom(get_value_dec("action", "null"), "EventValue").add_hints(type=str),
            opt_in=ne.custom(get_value_dec("opt_in", "null"), "EventValue").add_hints(type=bool)
        )

    gb_splash_action = splash_action_table\
        .groupby("splash_action_type", "opt_in")\
        .aggregate(count=na.count())\
        .map(get_action_no)

    enable_bar_table = squeeze_table\
        .filter(
            nf.equals("EventName", "searchlib_enable_bar")
        ).project(
            "UUID",
            "EventTimestamp",
            enable_type=ne.custom(get_value_dec("enable", "null"), "EventValue").add_hints(type=bool)
        )

    gb_splash_disable = splash_action_table\
        .join(
            enable_bar_table.filter(nf.equals("enable_type", False)),
            by=("UUID", "EventTimestamp"),
            type="inner"
        ).aggregate(splash_bar_disable=na.count())\
        .project(ne.all(), fielddate=ne.const(table_date))

    gb_splash_shown\
        .join(gb_splash_action, by="fielddate")\
        .join(gb_splash_disable, by="fielddate")\
        .publish(report, remote_mode=False)

    return job


if __name__ == "__main__":
    cli.run()

