-- Card util functions

$get_card_content_source_type = ($document) -> {
    RETURN Yson::LookupString($document.content[0], 'source_type');
};
$get_card_content_source_url = ($document) -> {
    RETURN Yson::LookupString($document.content[0].source, 'url');
};

$get_card_content_image_url = ($document, $prefix) -> {
    RETURN
        CAST(
            'https://avatars.mds.yandex.net/get-pdb/'
            || CAST(Yson::ConvertToInt64($document.content[0].content.group_id) as String)
            || '/'
            || Yson::ConvertToString($document.content[0].content.avatars_key)
            || $prefix
            AS String
        );
};

$get_card_source_meta_page_domain = ($document) -> {
    RETURN Yson::LookupString($document.source_meta, 'page_domain');
};
$get_card_source_meta_page_title = ($document) -> {
    RETURN Yson::LookupString($document.source_meta, 'title');
};
$get_card_source_meta_page_url = ($document) -> {
    RETURN Yson::LookupString($document.source_meta, 'page_url');
};

$get_card_meta_info_type = ($document) -> {
    RETURN Yson::LookupString($document.meta_info, 'type');
};
$get_card_meta_info_origin = ($document) -> {
    RETURN Yson::LookupString($document.meta_info, 'origin');
};

$get_card_page_meta_info_type = ($document) -> {
    RETURN Yson::LookupString($document.page_meta_info, 'type');
};

$get_card_origin_client_name = ($document) -> {
    RETURN Yson::LookupString($document.origin.client, 'name');
};

$get_card_origin_client_ui = ($document) -> {
    RETURN Yson::LookupString($document.origin.client, 'ui');
};

$get_card_origin_action = ($document) -> {
    RETURN Yson::LookupString($document.origin, 'action');
};

$get_card_origin_ui = ($document) -> {
    $card_origin_client_name = $get_card_origin_client_name($document);
    $card_origin_client_ui = $get_card_origin_client_ui($document);
    $is_card_origin_ui_yandexapp =
        String::Contains($card_origin_client_name, 'searchplugin')
        OR String::Contains($card_origin_client_name, 'ru.yandex.mobile')
        OR String::Contains($card_origin_client_name, 'yandexsearch');

    $card_origin_ui =
    CASE
        WHEN String::Contains($card_origin_client_ui, 'desktop') THEN 'desktop'
        WHEN $is_card_origin_ui_yandexapp THEN 'yandexApp'
        WHEN String::Contains($card_origin_client_ui, 'touch') THEN 'mobile'
        WHEN String::Contains($card_origin_client_ui, 'pad') THEN 'tablet'
        WHEN $card_origin_client_ui == 'common' THEN 'common'
        WHEN $card_origin_client_ui IS NULL AND ($card_origin_client_name == 'lego' OR $card_origin_client_name == 'yabrowser')
            THEN 'desktop'
        ELSE 'empty'
    END;

    RETURN $card_origin_ui;
};

$get_card_service_created_at = ($document) -> {
    RETURN Yson::LookupString($document.service.created_at, '_$date');
};
$get_card_service_updated_at = ($document) -> {
    RETURN Yson::LookupString($document.service.updated_at, '_$date');
};
$string_date_to_timestamp = ($str_date) -> {
    RETURN DateTime::MakeTimestamp(DateTime::ParseIso8601($str_date));
};
$get_card_service_created_at_ts = ($document) -> {
    RETURN $string_date_to_timestamp($get_card_service_created_at($document));
};
$get_card_service_created_at_date = ($document) -> {
    RETURN SUBSTRING($get_card_service_created_at($document), NULL, 10);
};

$get_card_description = ($document) -> {
    RETURN Yson::LookupString($document, 'description');
};

$is_card_private = ($document) -> {
    RETURN Yson::LookupBool($document, 'is_private');
};

$is_card_banned = ($document) -> {
    RETURN COALESCE(Yson::LookupBool($document, 'ban'), False);
};

$is_card_domain_market = ($document) -> {
    RETURN String::Contains($get_card_source_meta_page_domain($document), 'market.yandex');
};

$get_card_dashboard_type = ($document) -> {
    $goods_source_type = AsList('image', 'link', 'market', 'offer');

    $card_content_source_type = $get_card_content_source_type($document);
    $card_meta_info_type = $get_card_meta_info_type($document);
    $card_page_meta_info_type = $get_card_page_meta_info_type($document);
    $card_origin_client_name = $get_card_origin_client_name($document);

    $is_meta_info_type_goods = $card_meta_info_type == 'goods';
    $is_goods_card = $card_content_source_type == 'offer' -- offer -> goods
        OR $is_meta_info_type_goods 
            AND (
                $card_content_source_type IN $goods_source_type
                OR $card_content_source_type IS NULL
            )
    ;
    $is_image_card = $card_content_source_type == 'image';
    $is_unknown_goods_card = $card_origin_client_name IS NULL OR $card_origin_client_name == 'import_market_favorites';

    $dashboard_card_type = IF(
        $is_goods_card,
        IF(
            $is_image_card,
            'image-goods',
            IF(
                $is_unknown_goods_card,
                '???-goods',
                'goods'
            )
        ),
        COALESCE($card_content_source_type, '???-source_type')
    );

    RETURN $dashboard_card_type;
};


-- User util functions

$get_user_uid = ($document) -> {
    RETURN Yson::LookupString($document.uid, '_$numberLong');
};
$get_user_yandexuid = ($document) -> {
    RETURN Yson::LookupString($document, 'yandexuid');
};

$get_user_display_name = ($document, $get_private_data) -> {
    RETURN IF($get_private_data, Yson::LookupString($document, 'display_name'), Null);
};
$get_user_email = ($document, $get_private_data) -> {
    RETURN IF($get_private_data, Yson::LookupString($document, 'email'), Null);
};
$get_user_description = ($document, $get_private_data) -> {
    RETURN IF($get_private_data, Yson::LookupString($document, 'description'), Null);
};
$get_user_login = ($document, $get_private_data) -> {
    RETURN IF($get_private_data, Yson::LookupString($document, 'login'), Null);
};
$get_user_sex = ($document, $get_private_data) -> {
    RETURN IF($get_private_data, Yson::LookupString($document, 'sex'), Null);
};
$get_user_karma = ($document, $get_private_data) -> {
    RETURN IF($get_private_data, Yson::LookupString($document.karma, '_$numberLong'), Null);
};
$get_user_language = ($document, $get_private_data) -> {
    RETURN IF($get_private_data, Yson::LookupString($document, 'language'), Null);
};
$get_user_location = ($document, $get_private_data) -> {
    RETURN IF($get_private_data, Yson::Lookup($document, 'location'), Null);
};
$get_user_messenger_guid = ($document, $get_private_data) -> {
    RETURN IF($get_private_data, Yson::LookupString($document, 'messenger_guid'), Null);
};

-- Subqueries

DEFINE SUBQUERY $get_cards($start_date?, $end_date?, $use_fast_dump?, $cluster?) AS
    $cluster = $cluster ?? 'hahn';

    $use_fast_dump = $use_fast_dump ?? True;

    $cards_table_path = IF($use_fast_dump,
                           '//home/collections-backups/db/prod/collections-fast-dump/latest/card',
                           '//home/collections-backups/db/prod/podborki/latest/card'
                        );

    SELECT
        -- document,

        id AS card_id,
        board AS card_board,

        $get_card_content_source_type(document) AS card_content_source_type,
        $get_card_content_source_url(document) AS card_content_source_url,
        $get_card_content_image_url(document, '/orig') AS card_content_image_url,

        $get_card_source_meta_page_domain(document) AS card_source_meta_page_domain,
        $get_card_source_meta_page_title(document) AS card_source_meta_page_title,
        $get_card_source_meta_page_url(document) AS card_source_meta_page_url,

        $get_card_meta_info_type(document) AS card_meta_info_type,
        $get_card_meta_info_origin(document) AS card_meta_info_origin,
        $get_card_page_meta_info_type(document) AS card_page_meta_info_type,

        $get_card_service_created_at(document) AS card_service_created_at,
        $get_card_service_updated_at(document) AS card_service_updated_at,
        $get_card_service_created_at_ts(document) AS card_service_created_at_ts,
        $get_card_service_created_at_date(document) AS card_service_created_at_date,

        $get_card_origin_client_name(document) AS card_origin_client_name,
        $get_card_origin_client_ui(document) AS card_origin_client_ui,
        $get_card_origin_action(document) AS card_origin_action,
        $get_card_origin_ui(document) AS card_origin_ui,
        $get_card_description(document) AS card_description,

        $is_card_private(document) AS is_private,
        $is_card_banned(document) AS is_banned,
        $is_card_domain_market(document) AS is_card_domain_market,

        $get_card_dashboard_type(document) AS card_dashboard_type,

        owner AS card_owner_id
    FROM yt:$cluster.$cards_table_path
    WHERE
        (
            $start_date IS NULL
            OR $get_card_service_created_at(document) >= $start_date
        )
        AND
        (
            $end_date IS NULL
            OR $get_card_service_created_at(document) <= $end_date
        )
    ;
END DEFINE;

DEFINE SUBQUERY $get_redir_logs($start_date, $end_date, $cluster?) AS
    $cluster = $cluster ?? 'hahn';
    SELECT *
    FROM 
        yt:$cluster.RANGE(
            `home/logfeller/logs/collections-redir-log/1d`, 
            $start_date,
            $end_date
        )
    ;
END DEFINE;


DEFINE SUBQUERY $get_users($use_fast_dump?, $get_private_data?, $cluster?) AS
    $cluster = $cluster ?? 'hahn';

    $use_fast_dump = $use_fast_dump ?? True;
    $get_private_data = $get_private_data ?? False;

    $users_table_path = IF($use_fast_dump,
                           '//home/collections-backups/db/prod/collections-fast-dump/latest/user',
                           '//home/collections-backups/db/prod/podborki/latest/user'
                        );

    SELECT
        -- document,

        id AS user_id,

        $get_user_uid(document) AS user_puid,
        $get_user_yandexuid(document) AS yandexuid,

        $get_user_display_name(document, $get_private_data) AS display_name,
        $get_user_email(document, $get_private_data) AS email,
        $get_user_description(document, $get_private_data) AS description,
        $get_user_login(document, $get_private_data) AS login,
        $get_user_sex(document, $get_private_data) AS sex,
        $get_user_karma(document, $get_private_data) AS karma,
        $get_user_language(document, $get_private_data) AS language,
        $get_user_location(document, $get_private_data) AS location,
        $get_user_messenger_guid(document, $get_private_data) AS messenger_guid

    FROM yt:$cluster.$users_table_path
    ;
END DEFINE;

export $get_card_content_source_type;
export $get_card_content_source_url;
export $get_card_content_image_url;

export $get_card_source_meta_page_domain;
export $get_card_source_meta_page_title;
export $get_card_source_meta_page_url;

export $get_card_meta_info_type;
export $get_card_page_meta_info_type;

export $get_card_service_created_at;
export $get_card_service_updated_at;
export $string_date_to_timestamp;
export $get_card_service_created_at_ts;
export $get_card_service_created_at_date;

export $get_card_origin_client_name;
export $get_card_description;

export $is_card_domain_market;

export $get_card_dashboard_type;


export $get_user_uid;
export $get_user_yandexuid;

export $get_user_display_name;
export $get_user_email;
export $get_user_description;
export $get_user_login;
export $get_user_sex;
export $get_user_karma;
export $get_user_language;
export $get_user_location;
export $get_user_messenger_guid;


export $get_cards;
export $get_redir_logs;
export $get_users;
