#!/usr/bin/env python
# -*- coding: utf-8 -*-
from __future__ import division
import codecs
import os
import json
import copy
import argparse
from collections import defaultdict, Counter
import requests


class RandomGenerator(object):
    def __init__(self):
        self.cnt = 0

    def __call__(self):
        self.cnt += 1
        return "random://{}".format(self.cnt)


def avg(lst):
    try:
        return sum(lst) / len(lst)
    except ZeroDivisionError:
        return 0


API = "https://solomon.yandex.net/api/v2/push"


class SolomonPusher(object):
    def __init__(self, **kwargs):
        self.sensor_stub = {
            "labels": {
                "field": "value",
                "column_name": kwargs["column_name"],
                "measurement": "freon_video_{}_ru",
            },
            "ts": int(kwargs["timestamp"]),
            "value": "",
        }
        self.data = {
            "sensors": [],
            "commonLabels": {
                "project": "video",
                "cluster": "video_robot",
                "service": "legacy",
            },
        }
        self.sensors = []

    def add_point(self, **kwargs):
        sensor = copy.deepcopy(self.sensor_stub)
        sensor["labels"]["measurement"] = sensor["labels"][
            "measurement"
        ].format(kwargs["name"])
        sensor["value"] = str(round(kwargs["value"], 4))
        self.data["sensors"].append(sensor)

    def push(self):
        headers = {"Authorization": "OAuth {}".format(os.environ["SOLOMON_TOKEN"])}
        req = requests.post(
            API,
            json=self.data,
            verify=False,
            headers=headers,
            params=self.data["commonLabels"]
        )
        print(req.content)
        assert req.status_code == 200


rnq = RandomGenerator()


def pad_serp(urls, n):
    if len(urls) >= n:
        return urls[:n]
    res = copy.copy(urls)
    res += [rnq()] * (n - len(urls))
    return res


def dumpableize(obj):
    if (
        not isinstance(obj, dict)
        and not isinstance(obj, Counter)
        and not isinstance(obj, defaultdict)
    ):
        return obj
    return {format(k): dumpableize(v) for k, v in obj.items()}


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument("querygroups")
    parser.add_argument("--timestamp")
    parser.add_argument("--column_name")
    parser.add_argument("--output")
    parser.add_argument("--solomon_data")
    parser.add_argument("--debug", action="store_true")
    args = parser.parse_args()

    with open(args.querygroups) as f:
        qgs = json.load(f)

    total_urls_target = 100 * len(qgs)
    total_serps_target = 10 * len(qgs)
    total_urls = 0
    total_serps = 0

    cntr = defaultdict(lambda: defaultdict(Counter))
    values = []
    solomon = SolomonPusher(
        timestamp=args.timestamp, column_name=args.column_name
    )
    for qg in qgs:
        try:
            query = qg[0]["query"]
        except IndexError:
            # handle zero query
            values.append(
                {"p01": 0.0, "p05": 0.0, "p10": 0.0, "p05i": 0.0, "p10i": 0.0}
            )
        for req in qg:
            total_urls += len(req["serp"])
            total_serps += 1
            urls = [x["url"] for x in req["serp"]]
            cntr[query]["p01"][tuple(pad_serp(urls, 1))] += 1
            cntr[query]["p05"][tuple(pad_serp(urls, 5))] += 1
            cntr[query]["p10"][tuple(pad_serp(urls, 10))] += 1
            cntr[query]["p05i"][frozenset(pad_serp(urls, 5))] += 1
            cntr[query]["p10i"][frozenset(pad_serp(urls, 10))] += 1
        value = {
            "p01": cntr[query]["p01"].most_common(1)[0][1] / 10.0,
            "p05": cntr[query]["p05"].most_common(1)[0][1] / 10.0,
            "p10": cntr[query]["p10"].most_common(1)[0][1] / 10.0,
            "p05i": cntr[query]["p05i"].most_common(1)[0][1] / 10.0,
            "p10i": cntr[query]["p10i"].most_common(1)[0][1] / 10.0,
        }
        values.append(value)
    for p in ["p01", "p05", "p10", "p05i", "p10i"]:
        solomon.add_point(
            name="stability_vh_{}".format(p), value=avg([x[p] for x in values])
        )
    solomon.add_point(
        name="stability_vh_urls", value=total_urls / total_urls_target
    )
    solomon.add_point(
        name="stability_vh_serps", value=total_serps / total_serps_target
    )
    if not args.debug:
        solomon.push()
    with codecs.open(args.output, "w", "utf8") as f:
        json.dump(dumpableize(cntr), f, indent=2, ensure_ascii=False)
    with codecs.open(args.solomon_data, "w", "utf8") as f:
        json.dump(solomon.data, f, indent=2, ensure_ascii=False)


if __name__ == "__main__":
    main()
