#!/usr/bin/env python
# -*- coding: utf-8 -*-

import argparse
import json
import scipy as sp
import scipy.stats
import re


TABLE_STYLE = '<style>\n' \
              'table {border: 1px solid grey;text-align:center}' \
              'th {border: 1px solid grey;}' \
              'td {border: 1px solid grey;}\n' \
              '</style>\n'


COLOR = {'grey': '#CCCCCC', 'green': '#00FF00', 'lgreen': '#66FF66', 'llgreen': '#CCFFCC',
        'red': '#FF0000', 'lred': '#FF6666', 'llred': '#FFCCCC'}


PATTERN = '<iframe src='


def HandleOption():
    parser = argparse.ArgumentParser()
    parser.add_argument("--tasks", dest="tasks", help="Input file with toloker markup.", required=True)
    parser.add_argument("--honeypots", dest="honeypots", help="Input file with honeypots results.", required=True)
    parser.add_argument("--threshold", dest="threshold", help="Threshold to measure the tail with worst samples.", default=0.3)
    parser.add_argument("--report", dest="report", help="Output report.", required=True)
    return parser


def get_pval(wins, losses):
    pv = scipy.stats.binom_test(wins, wins + losses)
    return pv


def get_wins(bypair, left, right):
    assert(left != right)
    if (left, right) in bypair:
        return bypair[(left, right)]
    else:
        wins = bypair[(right, left)]
        return {'right_wins': wins['left_wins'], 'left_wins': wins['right_wins']}


def generate_bypair_table(systems, bypair):
    bypair_table = TABLE_STYLE + '<table>\n'
    # header
    bypair_table += '<tr>' + '<th></th>' + ('<th>%s</th>' * len(systems)) % tuple(systems) + '</tr>\n'
    for left in systems:
        bypair_table += '<tr>\n'
        bypair_table += '<th>%s</th>\n' % left
        for right in systems:
            if left != right:
                wins = get_wins(bypair, left, right)
                pval = get_pval(wins['left_wins'], wins['right_wins'])
                percentage = 100.0 * wins['left_wins'] / (wins['left_wins'] + wins['right_wins'])
                if wins['left_wins'] < wins['right_wins']:
                    if pval < 0.001:
                        color = 'red'
                        sign = '&lt&lt'
                    elif pval < 0.01:
                        color = 'lred'
                        sign = '&lt'
                    elif pval < 0.05:
                        color = 'llred'
                        sign = '&lt='
                    else:
                        color = 'grey'
                        sign = '~'
                elif wins['right_wins'] < wins['left_wins']:
                    if pval < 0.001:
                        color = 'green'
                        sign = '&gt&gt'
                    elif pval < 0.01:
                        color = 'lgreen'
                        sign = '&gt'
                    elif pval < 0.05:
                        color = 'llgreen'
                        sign = '&gt='
                    else:
                        color = 'grey'
                        sign = '~'
                else:
                    color = 'grey'
                    sign = '~'
                bypair_table += '<td style="background:%s">' \
                                '<font size="4"> <b>%.2f%%</b> </font>' \
                                '<br> <i>%s</i> <br>' \
                                '<font size="2"><i>p=%.4f</i></font> </td>' %\
                                (COLOR[color], percentage, left + sign + right, pval)
            else:
                bypair_table += '<td></td>'
        bypair_table += '</tr>\n'
    bypair_table += '</table><br>\n'
    return bypair_table


def generate_totalscore_table(systems, totalscore):
    totalscore_table = TABLE_STYLE + '<table>\n'
    # header
    totalscore_table += '<tr><th>Name</th><th>Wins percentage</th><th>Bad percentage</th><th>Wins</th><th>Comparisons</th></tr>\n'
    for sys in systems:
        wins, loss, bad = totalscore[sys]['win'], totalscore[sys]['loss'], totalscore[sys]['bad']
        wins_percentage = 100.0 * wins / (wins + loss)
        totalscore_table += '<tr><th>%s</th><td>%.2f%%</td><td>%.2f%%</td><td>%d</td><td>%d</td></tr>\n' %\
                            (sys, wins_percentage, bad, wins, wins + loss)
    totalscore_table += '</table><br>\n'
    return totalscore_table


def generate_by_screen_table(systems, by_screen):
    by_screen_table = TABLE_STYLE + '<table>\n'
    # header
    by_screen_table += '<tr><th>Video</th>'
    for sys in systems:
        by_screen_table += '<th>{}</th>'.format(sys)
    by_screen_table += '</tr>\n'
    for vid_name in by_screen.keys():
        by_screen_table += '<tr><th>{}</th>'.format(vid_name)
        for sys in systems:
            wins, loss = by_screen[vid_name][sys]['wins'], by_screen[vid_name][sys]['loss']
            url = by_screen[vid_name][sys]['url']
            comments = ''.join(['<br>' + comment for comment in by_screen[vid_name][sys]['comments']])
            if len(comments) > 0:
                comments = ' , ' + comments
            by_screen_table += '<td><a href="{}">{:.2f}%</a>{}</td>'.format(url, 100.0 * wins / (wins + loss), comments)
        by_screen_table += '</tr>\n'
    return by_screen_table


def get_hp_quality(data_hp):
    correct = 0.0
    for record in data_hp:
        gt = record['knownSolutions'][0]['outputValues']['side']
        vote = record['outputValues']['side']
        if gt == vote:
            correct += 1.0
    return correct, len(data_hp)


def get_videos_systems(data):
    vid_names = set()
    systems = set()
    for mark in data:
        vid_names.add(mark["inputValues"]["screen_name"])
        systems.add(mark["inputValues"]["sys_id_left"])
        systems.add(mark["inputValues"]["sys_id_right"])
    return vid_names, systems


def get_bypair(systems, by_screen_system):
    vid_names = by_screen_system.keys()
    bypair = {}
    for sys_left in systems:
        for sys_right in systems:
            pair = (sys_left, sys_right)
            rev_pair = (sys_right, sys_left)
            if pair not in bypair and rev_pair not in bypair:
                left_wins = sum([by_screen_system[vid_name][sys_left][sys_right]["score"] for vid_name in vid_names])
                right_wins = sum([by_screen_system[vid_name][sys_right][sys_left]["score"] for vid_name in vid_names])
                bypair[pair] = {'left_wins': left_wins, 'right_wins': right_wins}
    return bypair


def get_totalscore(systems, by_screen, threshold):
    totalscore = {}
    vid_names = by_screen.keys()
    for sys in systems:
        wins = sum([by_screen[vid_name][sys]["wins"] for vid_name in vid_names])
        losses = sum([by_screen[vid_name][sys]["loss"] for vid_name in vid_names])
        num_worst = len([vid_name for vid_name in vid_names if float(by_screen[vid_name][sys]["wins"]) /
                         (by_screen[vid_name][sys]["wins"] + by_screen[vid_name][sys]["loss"]) < threshold])
        totalscore[sys] = {"win": wins, "loss": losses, "bad": float(num_worst) / len(vid_names)}
    return totalscore


def get_by_screen(systems, by_screen_system):
    by_screen = {}
    vid_names = by_screen_system.keys()
    for vid_name in vid_names:
        line = {}
        for sys in systems:
            wins = sum([by_screen_system[vid_name][sys][sys_alt]['score'] for sys_alt in systems])
            loss = sum([by_screen_system[vid_name][sys_alt][sys]['score'] for sys_alt in systems])
            url = re.findall(PATTERN + '.* ', by_screen_system[vid_name][sys]['url'])[0][len(PATTERN): -1]
            comments = sum([by_screen_system[vid_name][sys_alt][sys].get('comments', []) for sys_alt in systems], [])
            line[sys] = {"wins": wins, "loss": loss, "url": url, "comments": comments}
        by_screen[vid_name] = line
    return by_screen


def collect_statistics(data, threshold):
    vid_names, systems = get_videos_systems(data)
    by_screen_system = {vid_name: {sys_left: {sys_right: {"score": 0, "comments": []}
                                              for sys_right in systems}
                                   for sys_left in systems}
                        for vid_name in vid_names}

    worker_tasks = {}
    times = []
    for mark in data:

        vid_name = mark["inputValues"]["screen_name"]
        pair = {"left": mark["inputValues"]["sys_id_left"], "right": mark["inputValues"]["sys_id_right"]}
        rev = {"left": "right", "right": "left"}
        winning_side = mark["outputValues"]["side"]
        if winning_side not in ["right", "left"]:
            continue
        sys_win = pair[winning_side]
        sys_loss = pair[rev[winning_side]]

        by_screen_system[vid_name][sys_win]["url"] = mark["inputValues"]["url_" + winning_side]
        by_screen_system[vid_name][sys_loss]["url"] = mark["inputValues"]["url_" + rev[winning_side]]
        by_screen_system[vid_name][sys_win][sys_loss]["score"] += 1
        if "comment" in mark["outputValues"]:
            by_screen_system[vid_name][sys_win][sys_loss]["comments"].\
                append(mark["outputValues"]["comment"])

        worker_tasks[mark['workerId']] = worker_tasks.get(mark['workerId'], 0) + 1
        times.append(mark['submitTs'] - mark['createTs'])

    stats = {'Number of workers': len(worker_tasks),
             'Number of tasks per worker (average)': sp.mean(worker_tasks.values()),
             'Number of tasks per worker (median)': sp.median(worker_tasks.values()),
             'Time per task (average)': sp.mean(times) / 1000.0,
             'Time per worker (median)': sp.median(times) / 1000.0}

    bypair = get_bypair(systems, by_screen_system)
    by_screen = get_by_screen(systems, by_screen_system)
    totalscore = get_totalscore(systems, by_screen, threshold)

    return bypair, totalscore, by_screen, stats


def main():
    args = HandleOption().parse_args()
    tasks_filename = args.tasks
    honeypots_filename = args.honeypots
    report_filename = args.report

    with open(tasks_filename, "r") as f_tasks:
        data = json.load(f_tasks)
    bypair, totalscore, by_screen, stats = collect_statistics(data, args.threshold)

    systems = sorted(
        totalscore,
        key=lambda x: float(totalscore[x]['win']) / (totalscore[x]['win'] + totalscore[x]['loss']),
        reverse=True)
    bypair_table = generate_bypair_table(systems, bypair)
    totalscore_table = generate_totalscore_table(systems, totalscore)
    by_screen_table = generate_by_screen_table(systems, by_screen)
    table = bypair_table + '<br>' + totalscore_table + '<br>' + by_screen_table

    with open(honeypots_filename, "r") as f_hp:
        data_hp = json.load(f_hp)
        hp_correct, num_honeypots = get_hp_quality(data_hp)
    if num_honeypots > 0:
        hp_quality = hp_correct * 100.0 / num_honeypots
    else:
        hp_quality = 'N/A'
    table += u"<p>Quality on honeypots = {}% (on {} honeypots)</p>".format(hp_quality, num_honeypots)

    stats_keys = sorted(stats.keys())
    for key in stats_keys:
        table += u"<p>{} = {:.2f}</p>".format(key, stats[key])

    with open(report_filename, "w") as output:
        output.write(table.encode('utf8'))


if __name__ == '__main__':
    main()
