# -*- coding: utf-8 -*-

from nile.api.v1 import (
    filters as nf,
    aggregators as na,
    extractors as ne,
    statface as ns,
    cli,
    with_hints,
    clusters,
    Record,
)

from qb2.api.v1 import typing as qt

import nile
import json
import math


output_schema_0 = {
    'uid': qt.String,
    'domain': qt.String,
    'ui': qt.String,
    'platform': qt.String,
    'is_related': qt.String,
    'from_wizard': qt.String,
    'not_found': qt.String,
    'tvt': qt.UInt64,
    'lvt': qt.Float,
}


@with_hints(output_schema=output_schema_0)
def parse_us(session):
    import libra
    import uatraits

    for key, rcont in session:
        requests = rcont.GetRequests()
        uid = key.replace('y', '').replace('uu/', '')
        for r in requests:
            if r.IsA('TYandexVideoRequest') or r.IsA('TYandexRelatedVideoRequest'):
                ui = 'desktop'
            elif r.IsA('TTouchYandexVideoRequest') or r.IsA('TTouchYandexRelatedVideoRequest'):
                ui = 'touch'
            elif r.IsA('TPadYandexVideoRequest') or r.IsA('TPadYandexRelatedVideoRequest'):
                ui = 'pad'
            elif r.IsA('TMobileAppYandexVideoRequest') or r.IsA('TMobileAppYandexRelatedVideoRequest'):
                ui = 'app'
            else:
                continue
            if (
                r.IsA('TYandexRelatedVideoRequest')
                or r.IsA('TTouchYandexRelatedVideoRequest')
                or r.IsA('TPadYandexRelatedVideoRequest')
                or r.IsA('TMobileAppYandexRelatedVideoRequest')
            ):
                is_related = True
            else:
                is_related = False

            try:
                detector = uatraits.detector('/usr/share/uatraits/browser.xml')
                detect_res = detector.detect(r.UserAgent)
                if 'isTablet' in detect_res and detect_res['isTablet']:
                    platform = "pad"
                elif 'isMobile' in detect_res and detect_res['isMobile']:
                    platform = "touch"
                else:
                    platform = "desktop"
            except:
                platform = "unknown"

            domain = r.ServiceDomRegion
            if not domain:
                domain = "unknown"
            domain = domain.replace(":443", "")

            full_request = r.FullRequest
            if "path=wizard" in full_request or "path=yandex_search" in full_request:
                from_wiz = True
            else:
                from_wiz = False

            if "/preview" in full_request:
                is_preview = True
            else:
                continue

            nf_counter = 0
            for block in r.GetBSBlocks():
                if "tech/viewer/not_found" in block.Path:
                    nf_counter += 1

            tvt = 0
            lvt = 0
            for bl in r.GetMainBlocks():
                results = bl.GetChildren()
                for result in results:
                    if not result.IsA("TVideoResultProperties"):
                        continue

                    url = str(result.Url)

                    duration = r.FindVideoDurationInfo(result)
                    if duration:
                        dt = min(duration.PlayingDuration, duration.Duration)
                        dur = max(duration.PlayingDuration, duration.Duration)
                    else:
                        dt = 0
                        dur = 0

                    heartbeat = r.FindVideoHeartbeat(result, 'ANY')
                    if heartbeat:
                        ht = heartbeat.Ticks
                    else:
                        ht = 0

                    dur = result.Duration

                    res_tvt = max(dt, ht)
                    tvt += res_tvt

                    if res_tvt > 30:
                        lvt = math.log(res_tvt, math.e)

                    if tvt > 0:
                        watched = 1
                    else:
                        watched = 0

            if is_preview:
                for fw in [str(from_wiz), "_total_"]:
                    for nf in [str(nf_counter > 0), "_total_"]:
                        for ir in [str(is_related), "_total_"]:
                            for u in [ui, "_total_"]:
                                for p in [platform, "_total_"]:
                                    for d in [domain, "_total_"]:
                                        yield Record(
                                            from_wizard=fw,
                                            not_found=nf,
                                            is_related=ir,
                                            ui=u,
                                            platform=p,
                                            domain=d,
                                            tvt=tvt,
                                            lvt=lvt,
                                            uid=uid,
                                        )


@cli.statinfra_job
def make_job(job, options, statface_client):

    job = job.env(
        yt_spec_defaults=dict(pool_trees=["physical"], use_default_tentative_pool_trees=True),
        templates=dict(job_root='//tmp'),
    )

    report = ns.StatfaceReport().path('Video.All/videowiz/stability_stats').scale('daily').client(statface_client)

    date = options.dates[0]

    processed = (
        job.table('user_sessions/pub/video/daily/%s/clean' % date)
        .libra(
            parse_us,
            libra_file=nile.files.RemoteFile("//statbox/resources/libra_nile_udf2.7.so"),
            memory_limit=5 * 1024,
        )
        .groupby('from_wizard', 'is_related', 'ui', 'not_found', 'platform', 'domain')
        .aggregate(
            requests=na.count(),
            tvt=na.sum('tvt'),
            lvt=na.sum('lvt'),
            auditory=na.count_distinct('uid'),
        )
        .project(ne.all(), fielddate=ne.const(date).with_type(str))
        .publish(report)
    )

    return job


if __name__ == '__main__':
    cli.run()
