# -*- coding: utf-8 -*-

from nile.api.v1 import (
    filters as nf,
    aggregators as na,
    extractors as ne,
    statface as ns,
    cli,
    with_hints,
    clusters,
    Record,
)

from qb2.api.v1.typing import *

import nile
import json
import math
import sys

output_schema_0 = {
    "ui": String,
    "path": String,
    "event": String,
    "sub_type": String,
    "platform": String,
    "from_block": String
}


@with_hints(output_schema=output_schema_0, ensure_optional_types=True)
def parse_us(sessions):
    for key, rcont in sessions:
        session = rcont.GetRequests()

        for request in session:
            if request.IsA("TYandexWebRequest"):
                ui = "desktop"
            elif request.IsA("TMobileYandexWebRequest") or request.IsA("TTouchYandexWebRequest"):
                ui = "touch"
            elif request.IsA("TPadYandexWebRequest"):
                ui = "pad"
            elif request.IsA("TMobileAppYandexWebRequest"):
                ui = "app"
            else:
                continue

            joiner = request.BaobabTree()
            for techEvent in request.GetYandexTechEvents():
                if joiner is not None:
                    baobabTech = techEvent.BaobabTech(joiner)
                    try:
                        techType = baobabTech.type
                    except:
                        techType = ""
                    if baobabTech is not None and techType == "widget-purchase":
                        data = baobabTech.data
                        sub_type = (data.get("subscriptionType", "Unknown"),)
                        platform = (data.get("platform", "Unknown"),)
                        from_block = (data.get("fromBlock", "Unknown"),)

                        path = data.get("type")
                        if not path:
                            continue
                        event = path.split("/")[1] if "/" in path else path

                        for u in [ui, "_total_"]:
                            for p in [platform, "_total_"]:
                                for st in [sub_type, "_total_"]:
                                    for fb in [from_block, "_total_"]:
                                        for ev in [event, "_total_"]:
                                            for pt in [path, "_total_"]:
                                                yield Record(
                                                    ui=u,
                                                    platform="".join(p),
                                                    sub_type="".join(st),
                                                    event=ev,
                                                    path=pt,
                                                    from_block="".join(fb),
                                                )


@cli.statinfra_job
def make_job(job, options, statface_client):
    job = job.env(
        templates={
            "job_root": "//tmp",
            "title": "New subscribers on SERP"
        },
        yt_spec_defaults={
            "pool_trees": ["physical"],
            "use_default_tentative_pool_trees": True,
        },
    )

    report = ns.StatfaceReport().path(
        "Video.All/plus/serp_subscription",
    ).scale(
        "daily",
    ).client(
        statface_client,
    )

    for date in options.dates:
        job.table(
            "user_sessions/pub/search/daily/%s/columns/clean" % date,
        ).libra(
            parse_us,
            memory_limit=4000,
            libra_file=nile.files.RemoteFile('//statbox/resources/libra_nile_udf2.7.so'),
        ).groupby(
            "ui",
            "path",
            "event",
            "sub_type",
            "platform",
            "from_block",
        ).aggregate(
            count=na.count(),
        ).project(
            ne.all(),
            fielddate=ne.const(date).with_type(String),
        ).publish(
            report,
        )

    return job


if __name__ == "__main__":
    cli.run()
