#-*- coding: UTF-8 -*-
import nile
import argparse
import time
from nile.api.v1 import (
    filters as nf,
    aggregators as na,
    extractors as ne,
    statface as ns,
    files as nfi,
    clusters,
    Record,
    cli,
    with_hints,
)
from qb2.api.v1 import (
    extractors as qe,
    filters as qf
)
from qb2.api.v1.logs import dsl as ql
from qb2.api.v1.typing import *
from copy import deepcopy
from datetime import datetime
import sys


REDIR_PREFIX_LIST = [
    "1412.1544.",
]

BLOCKS_STAT = {
    '1412': 'plus',
    '1544': 'popup',
    '1207': 'passport',
    '1478': 'cancel',
    '3511': 'order-success',
    '4023': 'tvod',
    '4024': 'est',
    '486': 'close',
    '487': 'show',
    '3513': 'frame-error',
    '3515': 'order-fail',
    '75': 'moretv',
    '3787': 'amedia',
    '791': 'after',
}

def counter_filter(path):
    if not path:
        return False
    for pre in REDIR_PREFIX_LIST:
        if path.startswith(pre):
            return True
    return False


@with_hints(output_schema=dict(platform=str,
                            event=str,
                            path=str,
                            from_block=str,
                            sub_type=str))
def prepare_to_stat(groups):
    import uatraits
    from datetime import datetime

    detector = uatraits.detector('/usr/share/uatraits/browser.xml')
    for key,recs in groups:

        platform = None
        from_block = None
        sub_type = None
        events = []
        for rec in recs:
            if not platform:
                platform = rec.get("ui", "Unknown")

            path = rec["path"]
            vars = rec.get("vars", "") or ""
            for v in vars.split(","):
                if not "=" in v:
                    continue
                vn = v.split("=")[0]
                vv = v.split("=")[1]
                if vn == "-subscriptionType":
                    sub_type = vv
                elif vn == "-location":
                    from_block = vv
                elif vn == "-platform":
                    pf = vn
            counter_name = '.'.join(map(lambda n: BLOCKS_STAT[n] if n in BLOCKS_STAT else n, path.split('.')))
            events.append(counter_name)

        for e in events:
            event = e.split(".")[2]
            for ui in [platform or "Unknown", "_total_"]:
                for fb in [from_block or "Unknown", "_total_"]:
                    for st in [sub_type or "Unknown", "_total_"]:
                        for path in [e, "_total_"]:
                            yield Record(
                                platform=ui,
                                event=event,
                                path=path,
                                from_block=fb,
                                sub_type=st,
                                )

@cli.statinfra_job
def make_job(job, options, statface_client):


    job = job.env(
        yt_spec_defaults=dict(
            pool_trees=["physical"],
            use_default_tentative_pool_trees=True
            ),
        templates=dict(
            tmp_root='//tmp',
            title='VideoSerpSubs'
            ),
        default_memory_limit=9000
        )

    report = ns.StatfaceReport() \
        .path('Video.All/plus/video_subs')\
        .scale('daily')\
        .client(statface_client)

    for date in options.dates:
        redir = job.table("logs/redir-log/1d/" + date)
        video_log = job.table("//home/videoquality/video_queries_cube/queries/" + date)
        video_related_log = job.table("//home/videoquality/video_queries_cube/queries.related/" + date)

        video_stream = job.concat(
            video_log,
            video_related_log,
            ).qb2(
                log='generic-yson-log',
                fields=[
                    qe.log_fields(
                        'ReqID',
                        'uid',
                        'ui',
                    ),
                ],
                filters=[
                    qf.defined('uid'),
                ],
                mode='records',
            )

        redir_events = redir.qb2(
                log='redir-log',
                fields=[
                    'yandexuid',
                    'path',
                    'reqid',
                    qe.log_field('vars'),
                ],
                filters=[
                    qf.defined('reqid', 'path', 'yandexuid'),
                    qf.custom(counter_filter, 'path'),
                ],
                mode='yamr_lines'
            ) \
            .join(video_stream, by_left="reqid", by_right="ReqID", type='left', assume_unique_right=True) \
            .filter(qf.defined('ReqID')) \
            .groupby('reqid') \
            .reduce(prepare_to_stat) \
            .groupby('platform', 'sub_type', 'event', 'path', 'from_block') \
                    .aggregate(
                               count=na.count(),
                               ) \
                    .project(ne.all(),
                             fielddate=ne.const(date).with_type(str)
                             ) \
            .publish(report, remote_mode=True)


    return job

if __name__ == '__main__':
    cli.run()
