#!/usr/bin/env python
# -*- coding: utf-8 -*-
from __future__ import division
import sys
import os
import codecs
import argparse
import datetime
import requests
import time
import gzip
import StringIO
import json
from collections import defaultdict, Counter
from pytils import counter_quantile, get_host
requests.packages.urllib3.disable_warnings()
from nile.api.v1 import statface as ns


headers = {
    'Authorization': 'OAuth {}'.format(os.environ['METRICS_TOKEN'])
}


def check_serpset(id_):
    retries = 0
    j = None
    while not j and retries < 3:
        try:
            j = requests.get(
                'https://metrics-calculation.qloud.yandex-team.ru'
                '/api/qex/metric-observation?regional=RU&evaluation=VIDEO'
                '&left-serp-set={s}&right-serp-set={s}'
                '&metric=video_nonplayable-5_share_touch'.format(s=id_),
                verify=False,
                headers=headers
            ).json()
        except Exception as e:
            retries += 1
            print('oops: {}'.format(e))
            time.sleep(5)
    for check in j[0]['checks']:
        if check['leftState'] != 'PASSED' or check['rightState'] != 'PASSED':
            return False
    return True


def get_date(id_):
    retries = 0
    j = None
    while not j and retries < 3:
        try:
            j = requests.get(
                'http://metrics.yandex-team.ru/services/api/serpset/info'
                '?id={s}'.format(s=id_),
                verify=False,
                headers=headers
            ).json()
        except Exception as e:
            retries += 1
            print('oops: {}'.format(e))
            time.sleep(5)
    return datetime.datetime.strptime(
        j['date'].split('T')[0], '%Y-%m-%d'
    ).date()


def get_serps(id_):
    retries = 0
    j = None
    while not j and retries <= 3:
        try:
            j = requests.get(
                'https://metrics-calculation.qloud.yandex-team.ru/api/json/'
                '{}?regional=RU&evaluation=VIDEO&aspect=availability_mobile'
                '&absolute=false&serpset-filter=onlySearchResult'.format(id_),
                verify=False,
                headers=headers
            )
        except Exception as e:
            print('oops: {}'.format(e))
            retries += 1
            time.sleep(5)
    try:
        return json.load(gzip.GzipFile(fileobj=StringIO.StringIO(j.content)))
    except:
        raise Exception(j.content)


def parse_date(s):
    return datetime.datetime.strptime(s, '%Y-%m-%d').date()


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument('date')
    parser.add_argument('--stat_login', '-sl', default='robot_pecheny')
    parser.add_argument('--stat_password', '-sp', required=True)
    args = parser.parse_args()

    args.date = parse_date(args.date)

    serpsets = requests.get(
        'https://metrics.yandex-team.ru/services/api/serpset/list/RU/VIDEO/'
        '?from={}T00:00:00.000&to={}T00:00:00.000'
        '&cronSerpDownloadId=9544'.format(
            args.date - datetime.timedelta(days=31),
            args.date - datetime.timedelta(days=1)
        ), verify=False
    ).json()

    print('getting serpsets')
    good_serpsets = [
        serpset for serpset in serpsets if check_serpset(serpset)
    ]

    print('getting dates')
    dates = {
        get_date(id_): id_ for id_ in good_serpsets
    }

    print('getting serps')
    d = defaultdict(dict)
    for date in sorted(dates):
        id_ = dates[date]
        serpset = get_serps(id_)
        print('processing serpset {}'.format(id_))
        for serp in serpset:
            if 'components' not in serp:
                continue
            for comp in serp['components']:
                if 'judgements.factor_vdp' not in comp:
                    continue
                result = comp['judgements.factor_vdp']['name']
                d[comp['componentUrl']['pageUrl']][date] = result

    length_counter = Counter()
    debug = defaultdict(list)
    k = d.keys()
    bk = [x for x in k if set(d[x].values()) == {'OK', 'BAD'}]
    for k in bk:
        keys = sorted(d[k].keys(), reverse=True)
        if d[k][keys[0]] == 'OK':
            continue
        last_day = keys[0] + datetime.timedelta(days=1)
        previous_day = None
        for i, e in enumerate(keys):
            if d[k][e] == 'OK':
                length_counter[(last_day - previous_day).days] += 1
                debug[(last_day - previous_day).days].append((k, d[k]))
                break
            previous_day = e

    for pair in length_counter.most_common():
        print('{:02}\t{}'.format(*pair))

    result = {
        'fielddate': args.date.strftime('%Y-%m-%d'),
        'platform': 'desktop',
        'host_category': '_total_'
    }
    for q in [0.1 * x for x in range(10)]:
        result['q{}'.format(int(q * 100))] = counter_quantile(
            length_counter, q
        )
        print('{:.03}: {}'.format(q, counter_quantile(length_counter, q)))

    host_counter = Counter()

    for x in debug:
        if x < 10:
            continue
        for pair in debug[x]:
            host_counter[get_host(pair[0])] += 1

    for pair in host_counter.most_common():
        print('{}\t{}'.format(*pair))

    client = ns.StatfaceClient(
        proxy='upload.stat.yandex-team.ru',
        username=args.stat_login,
        password=args.stat_password
    )

    ns.StatfaceReport().path(
        'Video/Others/vdp-lifetime'
    ).scale(
        'daily'
    ).client(
        client
    ).data(
        [result]
    ).publish()


if __name__ == "__main__":
    main()
