#!/usr/bin/env python
# -*- coding: utf-8 -*-
from __future__ import division
import datetime
import sys
import smtplib
import argparse
from collections import defaultdict
from email.mime.text import MIMEText
from videolog_common import get_cluster, get_driver
from nile.api.v1 import clusters


def parse_mtime(s):
    return datetime.datetime.strptime(
        s.replace("Z", ""), "%Y-%m-%dT%H:%M:%S.%f"
    ) + datetime.timedelta(hours=3)


class ErrorWrapper(object):
    def __init__(self):
        self.errors = []
        self.smtp = smtplib.SMTP("outbound-relay.yandex.net")

    def write(self, msg):
        if not msg.endswith("\n"):
            msg += "\n"
        sys.stderr.write(msg)
        self.errors.append(msg)

    def _send_email(self, addressees, msg):
        msg = MIMEText(msg)
        msg["Subject"] = "Production tables monitoring"
        msg["From"] = "prod-tables-watch@yandex-team.ru"
        msg["To"] = ",".join(addressees)
        print("Sending message to {}".format(addressees))
        self.smtp.sendmail(msg["From"], addressees, msg.as_string())

    def send_email(self, *addressees):
        text = "The following problems occurred:\n\n{}".format(
            "\n".join(self.errors)
        )
        self._send_email(addressees, text)


DEFAULT_ADDRESSEES = ("pecheny@yandex-team.ru", "ensuetina@yandex-team.ru")


tables = [
    {"cluster": "hahn", "table": "//home/videolog/strm_meta/iron_branch/concat"},
    {"cluster": "arnold", "table": "//home/videolog/strm_meta/iron_branch/concat"},
    {"cluster": "hahn", "table": "//home/videoindex/vhs/docbase/dynamic/direct_index", "addressees": ("pecheny@yandex-team.ru", "vika-pavlova@yandex-team.ru")},
    {"cluster": "hahn", "table": "//home/video-hosting/adv_tags/tags_data", "addressees": ("pecheny@yandex-team.ru", "vika-pavlova@yandex-team.ru")},
]


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument('--debug', action="store_true")
    args = parser.parse_args()

    cs = {}
    for cl in ["hahn", "arnold"]:
        cs[cl] = get_cluster(clusters, {"proxy": cl.title()})

    now = datetime.datetime.now()
    err_dict = defaultdict(lambda: ErrorWrapper())

    for table_ in tables:
        cl = table_["cluster"]
        table = table_["table"]
        addressees = table_.get("addressees") or DEFAULT_ADDRESSEES
        err = err_dict[addressees]
        driver = get_driver(cs[cl])
        print("checking table {} on cluster {}".format(table, cl))
        mod_time = driver.get_attribute(table, "modification_time")
        try:
            parsed = parse_mtime(mod_time)
        except (ValueError, TypeError):
            err.write("ERROR: broken mod_time on table {}:{} - {}".format(
                cl, table, mod_time
            ))
            continue
        delta = now - parsed
        totalsecs = delta.total_seconds()
        print("delta is {:.2} hours".format(totalsecs / 3600.0))
        if totalsecs >= 3600 * 10:
            err.write("ERROR: table {}:{} too old ({} hours)".format(
                cl, table, round(totalsecs / 3600.0, 2)
            ))
    for addressees in err_dict:
        err = err_dict[addressees]
        if err.errors or args.debug:
            err.send_email(*addressees)


if __name__ == "__main__":
    main()
