#-*- coding: UTF-8 -*-
import argparse
from datetime import datetime, timedelta
import codecs
import json

class RemapTable:
    def __init__(self, points):
        points_len = len(points)
        self.power = 0
        for v in points:
            if points_len + 1 > 1 << self.power:
                self.power += 1
            else:
                break

        elem_count = 1 << self.power

        self.CP = [points[-1]] * elem_count
        self.CP[0] = -1e20 # not used actually
        for i in xrange(0, points_len):
            self.CP[i + 1] = points[i]

        self.scale_data = [0] * (elem_count * 2)
        self.scale_data[0] = 0 # add
        self.scale_data[1] = 0 # mult
        for i in xrange(1, points_len):
            alpha = 1.0 / (points_len - 1)
            diff = 1.0 / (self.CP[i + 1] - self.CP[i])
            self.scale_data[2 * i] = (i - 1 - self.CP[i] * diff) * alpha # add
            self.scale_data[2 * i + 1] = diff * alpha # mult

        for i in xrange(points_len, elem_count):
            self.scale_data[2 * i] = 1 # add
            self.scale_data[2 * i + 1] = 0 # mult

    def remap(self, value):
        nCP = 0
        for i in xrange(self.power - 1, -1, -1):
            if value > self.CP[nCP + (1 << i)]:
                nCP += 1 << i
        return value * self.scale_data[nCP * 2 + 1] + self.scale_data[nCP * 2]

LVT_BY_7_DAYS_REMAP_TABLE = RemapTable([0,17.8241,35.9346,50.3469,64.9889,82.437,103.858,130.936,166.065,211.394,271.976,357.965,489.916,718.395,1171.82,2435.93,1.03507e+08])

def calc_and_remap_popularity(stats, timestamp, days_count=7):
    date = datetime.fromtimestamp(timestamp)
    end_date = date - timedelta(days=1)
    end_date_str = end_date.strftime('%Y-%m-%d')
    start_date = end_date - timedelta(days=days_count-1)
    start_date_str = start_date.strftime('%Y-%m-%d')

    lvt = 0.
    for date in stats:
        if date >= start_date_str and date <= end_date_str:
            lvt += stats[date]["lvt"]
    return LVT_BY_7_DAYS_REMAP_TABLE.remap(lvt)

def main():
    parser = argparse.ArgumentParser()
    parser.add_argument('--input', required=True)
    parser.add_argument('--output', required=True)
    args = parser.parse_args()

    with codecs.open(args.input, "r", "utf8") as inp:
        data = json.load(inp)
    results = []
    for url_info in data:
        url = url_info["cano_url"]
        joined = int(url_info["joined"])
        stats = url_info.get("data", {})
        for ts_milli in url_info["ts_list"]:
            to_append = {"url" : url, "joined" : joined}
            timestamp = ts_milli / 1000
            to_append["value"] = calc_and_remap_popularity(stats, timestamp)
            to_append["ts"] = ts_milli
            results.append(to_append)

    with codecs.open(args.output, "w", "utf8") as out:
        json.dump(results, out)

if __name__ == '__main__':
    main()
