import argparse
import json
import requests
from requests.exceptions import RequestException
import sys
import time


CONTENT_API_TEMPLATE = "https://frontend.vh.yandex.ru/v23/player/{}.json"


def get_with_retries(address, retry_count=10, sleep_time=10):
    for retry_index in xrange(retry_count):
        try:
            if retry_index > 0:
                sys.stderr.write("Retry #{}\n".format(time.time(), retry_index))
            response = requests.get(address)
            response.raise_for_status()
            return response
        except RequestException as re:
            sys.stderr.write("Exception occured inside get_with_retries: {}\n".format(re))
            time.sleep(sleep_time)
    return None


def get_content_thumbnail_orig(uuid):
    content_address = CONTENT_API_TEMPLATE.format(uuid)
    response = get_with_retries(content_address)
    try:
        return json.loads(response.content)["content"]["thumbnail"]
    except Exception as e:
        sys.stderr.write("Exception occured while getting thumbnail from json: {}\n".format(e))
        return None


def get_thumbnail_link(uuid, max_width, max_height):
    try:
        thumbnail = get_content_thumbnail_orig(uuid)
        if thumbnail:
            resize_param = '{}x{}'.format(max_width, max_height)
            thumbnail_link = "https:{}".format(thumbnail).replace("orig", resize_param)
            if get_with_retries(thumbnail_link):
                return thumbnail_link
    except Exception as e:
        sys.stderr.write('Exception occured while getting thumbnail: "{}"\n'.format(e))

    return None


if __name__ == '__main__':
    parser = argparse.ArgumentParser()
    parser.add_argument('--uuid', type=str, required=True)
    parser.add_argument('--max_width', type=int, required=True)
    parser.add_argument('--max_height', type=int, required=True)
    parser.add_argument('--output', type=str, required=True)
    args = parser.parse_args()

    result = {}
    if args.uuid:
        thumbnail_link = get_thumbnail_link(args.uuid, args.max_width, args.max_height)
        if thumbnail_link:
            result["image"] = "\"" + thumbnail_link + "\""

    with open(args.output, "w") as out:
        json.dump(result, out)
