#-*- coding: UTF-8 -*-
import argparse
import codecs
from nile.api.v1 import (
    clusters,
    Record
)
from datetime import datetime
from time import sleep
import json
import requests
import random

CLONE_GRAPH_TEMPLATE = "https://nirvana.yandex-team.ru/api/public/v1/cloneWorkflow?workflowId={}&newName={}&newQuotaProjectId={}"
SET_GRAPH_PARAMS = "https://nirvana.yandex-team.ru/api/public/v1/setGlobalParameters"
START_GRAPH_TEMPLATE = "https://nirvana.yandex-team.ru/api/public/v1/startWorkflow?workflowId={}"

UIDS_TABLE_DIRECTORY = "//home/ether_prod/pushes/spec_events"
TIME_FORMAT = '%Y-%m-%dT%H:%M:%S'
TIME_FORMAT_UTC = TIME_FORMAT + '+03:00'

QUOTA_ETHER_DEV = "ether-dev"

def get_push_time(ts):
    return datetime.fromtimestamp(ts).strftime(TIME_FORMAT_UTC)

def get_special_event_push_info(special_event_info, special_events_pushes_info, time, uuid):
    push_info = {}

    push_info["title"] = special_event_info["push_info"]["title"]
    push_info["body"] = special_event_info["push_info"]["body"]
    if uuid:
        push_info["uuid"] = uuid
    if special_event_info["push_info"].get("url"):
        push_info["custom_push_url"] = "\"" + special_event_info["push_info"]["url"] + "\""
    if special_event_info["push_info"].get("time"):
        push_info["push_time_moscow"] = special_event_info["push_info"]["time"] + "+03:00"
    else:
        push_info["push_time_moscow"] = time

    if "topic_push" in special_event_info["push_info"]:
        push_info["topic_push"] = special_event_info["push_info"]["topic_push"]
    if "image" in special_event_info["push_info"]:
        push_info["image"] = special_event_info["push_info"]["image"]

    push_info["from_block_prefix"] = special_events_pushes_info["from_block_prefix"]
    table_name_inner_part = push_info["from_block_prefix"]
    if special_event_info["push_info"].get("from_block_suffix"):
        push_info["from_block"] = special_event_info["push_info"]["from_block_suffix"]
        table_name_inner_part += push_info["from_block"]

    if special_event_info["push_info"].get("push_type"):
        push_info["push_type"] = special_event_info["push_info"]["push_type"]
    if special_event_info["push_info"].get("stat_id"):
        push_info["stat_id"] = special_event_info["push_info"]["stat_id"]
        table_name_inner_part += push_info["stat_id"]

    push_info["uids_table"] = UIDS_TABLE_DIRECTORY + push_info["push_time_moscow"][:10] + "_" + table_name_inner_part + "_" + str(random.randint(1, 10 ** 9))

    if special_event_info.get("query_words"):
        push_info["queries"] = ["\"" + query + "\"" for query in special_event_info["query_words"]]
        push_info["titles"] = push_info["queries"]
    if special_event_info["push_info"].get("auto_channels", []):
        push_info["tv_online_channels"] = [ "\"" + channel + "\"" for channel in special_event_info["push_info"]["auto_channels"]]
    if special_event_info["push_info"].get("programs", []):
        push_info["tv_online_programs"] = [ "\"" + program + "\"" for program in special_event_info["push_info"]["programs"]]
    if special_event_info["push_info"].get("object_ids", []):
        push_info["object_ids"] = ["\"" + object_id + "\"" for object_id in special_event_info["push_info"]["object_ids"]]
    if special_event_info.get("website"):
        push_info["urls"] = ["\"" + special_event_info["website"] + "\""] + ["\"" + url + "\"" for url in special_event_info.get("related_sites", [])]

    if special_event_info.get("crypta_info"):
        crypta_info = special_event_info["crypta_info"]

        for param in ["audience_segments",
                      "heuristic_internal",
                      "marketing_segments",
                      "heuristic_segments",
                      "heuristic_common",
                      "longterm_interests",
                      "lal_internal",
                      "lal_common"]:
            values = crypta_info.get(param)
            if values:
                push_info[param] = [str(elem) for elem in values]

    if special_event_info.get("direct_uids"):
        push_info["direct_uids"] = special_event_info.get("direct_uids")

    base_fetch_settings = special_events_pushes_info["base_fetch_settings"]
    for param in ["queries_threshold",
                  "object_ids_threshold",
                  "tv_online_channels_tvt_threshold",
                  "tv_online_programs_tvt_threshold",
                  "urls_threshold",
                  "titles_threshold"]:
        if base_fetch_settings.get(param):
            push_info[param] = base_fetch_settings[param]
        if special_event_info["push_info"].get(param):
            push_info[param] = special_event_info["push_info"][param]

    if special_event_info.get("send"):
        push_info["send"] = special_event_info["send"]

    return push_info

def find_special_event_push_info(config, time, uuid):
    special_events_pushes_info = config["special_events_pushes_info"]

    for special_event_info in config["special_events"]:
        if uuid and special_event_info["push_info"]["uuid"] != uuid:
            continue

        return get_special_event_push_info(special_event_info, special_events_pushes_info, time, uuid)
    return {}

def main():
    parser = argparse.ArgumentParser()
    parser.add_argument('--table', type=str, required=True)
    parser.add_argument('--date_from', type=str, required=True)
    parser.add_argument('--date_to', type=str, required=True)
    parser.add_argument('--base_workflow_id', type=str, required=True)
    parser.add_argument('--nirvana_token', type=str, required=True)
    parser.add_argument('--config', type=str, required=True)
    parser.add_argument('--quota_id', type=str, default=QUOTA_ETHER_DEV)
    args = parser.parse_args()

    headers = {'Content-Type': 'application/json',
               'Authorization': 'OAuth {}'.format(args.nirvana_token)}

    with codecs.open(args.config, 'r', 'utf8') as inp:
        config = json.load(inp)

    cluster = clusters.yt.Arnold()

    push_infos = []
    for rec in cluster.driver.read(args.table):
        if "deleted" in rec["YaTvEpisode.Options"]:
            continue
        push_info = find_special_event_push_info(config, get_push_time(int(rec["start_time_ts"])), rec["YaTvEpisode.UUID"])
        if not push_info:
            continue
        push_infos.append(push_info)

    date_from = datetime.strptime(args.date_from, "%Y-%m-%d")
    date_to = datetime.strptime(args.date_to, "%Y-%m-%d")
    special_events_pushes_info = config["special_events_pushes_info"]
    for special_event_info in config["special_events"]:
        if special_event_info.get("is_classic_special_event", True) == False:
            time = datetime.strptime(special_event_info["push_info"]["time"], TIME_FORMAT)
            if time < date_from or time > date_to:
                continue
            push_time = special_event_info["push_info"]["time"] + "+03:00"
            push_info = get_special_event_push_info(special_event_info, special_events_pushes_info, push_time, special_event_info["push_info"].get("uuid", ""))
            push_infos.append(push_info)

    for index, push_info in enumerate(push_infos[::-1]):
        if index > 0:
            sleep(1 * 60)

        ## Clone graph
        r = None
        while r is None or r.status_code != 200:
            r = requests.get(
                CLONE_GRAPH_TEMPLATE.format(
                    args.base_workflow_id,
                    (push_info["push_time_moscow"][:10] + " " + push_info["title"]).encode('utf8'),
                    args.quota_id,
                ),
                headers=headers,
                verify=False)
        print r.json(), r.status_code
        graph_id = r.json()['result']
        print "New workflow: ", "https://nirvana.yandex-team.ru/flow/{}".format(graph_id)

        ## Set params
        params = [{'parameter': elem, 'value': push_info[elem]} for elem in push_info]
        r = None
        while r is None or r.status_code != 200:
            r = requests.post(
                SET_GRAPH_PARAMS,
                data=json.dumps({
                    'jsonrpc': "2.0",
                    'id': 'id',
                    'method': 'setGlobalParameters',
                    'params': {'workflowId': graph_id,
                               'params': params}
                }),
                verify=False,
                headers=headers,
            )
        print "Set params", r.json(), r.status_code

        ## Run
        r = None
        while r is None or r.status_code != 200:
            r = requests.get(START_GRAPH_TEMPLATE.format(graph_id), headers=headers, verify=False)
        print "Start graph", r.json(), r.status_code, "\n"

if __name__ == '__main__':
    main()
