# -*- encoding: utf8 -*-
from common import *

import argparse
import json
import requests
import sys
import time

ICON = 'https://yastatic.net/s3/home/stream/push_pictures/icon_for_push.png'
NHL_ICON = 'https://storage.mds.yandex.net/get-sport/40184/b05c8d62-fed9-4cab-9f22-72cec4468d2d.png'
ETHER_TEMPLATE = 'https://{}.ru/efir?stream_id={}&from_block={}'
CONTENT_API_TEMPLATE = 'https://frontend.vh.yandex.ru/v23/player/{}.json'


def is_nhl(topic_push):
    return topic_push == "efir_nhl"


def make_sup_request(receiver, stream_id, title, body, push_id, thumbnail, ttl, topic_push="stream_push"):
    domain = "yandex"
    icon = ICON
    if is_nhl(topic_push):
        icon = NHL_ICON
        domain = "yandexsport"
    sys.stderr.write(domain + " : " + topic_push + " : " + stream_id + "\n")

    url = ETHER_TEMPLATE.format(domain, stream_id, push_id)
    sup_request = {}
    sup_request['receiver'] = receiver
    sup_request['ttl'] = ttl
    sup_request['schedule'] = 'now'
    sup_request['project'] = 'stream'
    sup_request['browser_features'] = {
        'requireInteraction': True
    }
    deep_update(sup_request, MOBILE_SILENT_SETTINGS)
    if thumbnail:
        deep_update(sup_request, {
            'android_features': {'image': thumbnail},
            'browser_features': {'image': thumbnail},
        })

    # if self.sup_backend != 'push-beta.n.yandex-team.ru':
    #     sup_request['throttle_policies'] = {
    #         'device_id' : 'efir_subscription_device_id',
    #         'install_id' : 'efir_subscription_install_id',
    #         "content_id": 'efir_subscription_content_id'
    #     }

    #sup_request['transport'] = 'Xiva'

    sup_request['data'] = {
        'push_id': push_id,
        'content_id': stream_id,
        'transit_id': stream_id,
        'push_action': 'uri',
        'push_uri': url,
        'topic_push': topic_push
    }
    sup_request['notification'] = {
        'title': title,
        'body': body,
        'icon': icon,
        'link': url
    }
#    sup_request["max_expected_receivers"] = 10

    return json.dumps(sup_request, ensure_ascii=False).encode('utf-8')


def test_stream_id(uuid):
    url = CONTENT_API_TEMPLATE.format(uuid)
    try:
        r = requests.get(url)
        r.raise_for_status()
    except requests.HTTPError as http_error:
        return repr(http_error)
    except requests.exceptions.ConnectionError as conn_err:
        return repr(conn_err)

    return None


def get_prepared_and_unprepared_documents(documents_for_check, args):
    documents_for_push = []
    unprepared_documents = []

    title_template = args.title.decode('utf8')
    body_template = args.body_template.decode('utf8')

    documents_for_check.sort(key=lambda rec: rec["uuid"])
    documents_for_check.append({"uuid": None})

    prev_uuid = documents_for_check[0]["uuid"] if documents_for_check else None
    recs_same_uuid = []

    valid_recs = {}

    for rec in documents_for_check:
        if rec["uuid"] == prev_uuid:
            recs_same_uuid.append(rec)
        else:
            active_rec = None
            for doc in recs_same_uuid:
                if doc["active"]:
                    active_rec = doc
            if not active_rec:
                unprepared_documents.append(recs_same_uuid[0])
            else:
                if not active_rec["blogger_id"] or (
                        not active_rec["uuid"].startswith("v") and not active_rec["output_stream"]):
                    unprepared_documents.append(active_rec)

                    prev_uuid = rec["uuid"]
                    recs_same_uuid = [rec]
                    continue

                error_text = test_stream_id(active_rec["uuid"])

                if error_text is None:
                    if active_rec["blogger_id"] not in valid_recs:
                        valid_recs[active_rec["blogger_id"]] = []

                    valid_recs[active_rec["blogger_id"]].append(active_rec)
                else:
                    rec_as_dict = active_rec.to_dict()

                    errors = []
                    if "errors" in rec_as_dict:
                        errors = json.loads(rec_as_dict["errors"])
                    errors.append({
                        "timestamp": int(time.time()),
                        "error": error_text,
                    })

                    sys.stderr.write(active_rec["uuid"] + " : " + error_text + "\n")

                    rec_as_dict["errors"] = json.dumps(errors)
                    unprepared_documents.append(Record(**rec_as_dict))

        prev_uuid = rec["uuid"]
        recs_same_uuid = [rec]

    for key, value in valid_recs.items():
        value = sorted(value, key = lambda rec: rec['release_date'], reverse=True)
        active_rec = value[0]

        receiver = ["tag:videoEvent=='new_theme_{}'".format(active_rec["blogger_id"])]
        title = title_template.format(active_rec["publisher_name"].decode('utf8'))
        if len(value) == 1:
            body = body_template.format(active_rec["title"].decode('utf8'))
        else:
            remBig = (len(value) / 10) % 10
            remSmall = len(value) % 10
            if remBig == 1 or remSmall == 0 or remSmall >= 5:
                body = ('Смотрите ' + str(len(value)) + ' новых выпусков').decode('utf8')
            elif remSmall == 1:
                body = ('Смотрите ' + str(len(value)) + ' новый выпуск').decode('utf8')
            else:
                body = ('Смотрите ' + str(len(value)) + ' новых выпуска').decode('utf8')

        thumbnail_url = active_rec["thumbnail_url"]
        if thumbnail_url:
            prefix = ""
            if not thumbnail_url.startswith("https:"):
                prefix = "https:"
            # Ends with "/orig" so we replace last 4 symbols with thumbnail_size (e.g. "1280x720").
            thumbnail_url = prefix + thumbnail_url[:-4] + args.image_size

        sup_request = make_sup_request(receiver, active_rec["uuid"], title, body, args.from_block, thumbnail_url, args.ttl)
        documents_for_push.append(Record(push=sup_request))

    return documents_for_push, unprepared_documents


def parse_arguments():
    parser = argparse.ArgumentParser()
    parser.add_argument('--documents', type=str, required=True)
    parser.add_argument('--push_table', type=str, required=True)
    parser.add_argument('--old_unprepared_documents', type=str, required=True)
    parser.add_argument('--new_unprepared_documents', type=str, required=True)
    parser.add_argument('--title', type=str, required=True)
    parser.add_argument('--body_template', type=str, required=True)
    parser.add_argument('--ttl', type=int, required=True)
    parser.add_argument('--from_block', type=str, required=True)
    parser.add_argument('--image_size', type=str, required=True)

    return parser.parse_args()


def main(args):
    documents_table = get_mr_table(args.documents)

    cluster = clusters.yt.Hahn(pool="ether_prod") \
        .env(parallel_operations_limit=10,
             yt_spec_defaults=dict(
                 pool_trees=["physical"],
                 tentative_pool_trees=["cloud"],
             ),
             templates=dict(
                 tmp_root=YT_ETHER_TMP_SUBSCRIBE_PUSH_FOLDER,
                 title='BloggerNewContentRecommendations',
             ))

    documents_for_check = []
    for rec in cluster.driver.read(documents_table):
        documents_for_check.append(rec)

    documents_for_push, unprepared_documents = get_prepared_and_unprepared_documents(documents_for_check, args)

    cluster.driver.write(args.new_unprepared_documents, unprepared_documents)
    cluster.driver.write(args.push_table, documents_for_push)

    cluster.driver.client.link(target_path=args.new_unprepared_documents,
                               link_path=args.old_unprepared_documents,
                               force=True)


if __name__ == '__main__':
    main(parse_arguments())
