#!/usr/bin/env python
# -*- coding: utf-8 -*-
import os
import math
import argparse
from requests import Session
from tqdm import tqdm
from staff_data import retry

START_URL = "https://staff-api.yandex-team.ru/v3/departmentstaff"


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument("--input", default="all_staff_data.tsv")
    parser.add_argument("--output", default="department_staff.tsv")
    args = parser.parse_args()

    with open(args.input) as f:
        cnt = f.read()
    cnt = cnt.split("\n")[:-1]
    cnt = [x.split("\t") for x in cnt]
    cnt = [dict(zip(cnt[0], x)) for x in cnt[1:]]
    chiefs = set()
    for x in cnt:
        for ch in x["chiefs"].split(","):
            chiefs.add(ch)
    chiefs = sorted(chiefs)
    chief_to_level = {}

    session = Session()
    session.headers.update({"Authorization": f"OAuth {os.environ['STAFF_TOKEN']}"})
    params = {"_fields": "person.login,role,department_group.level"}
    expected = math.ceil(len(chiefs) / 10.0)
    for _ in tqdm(range(expected)):
        batch, chiefs = chiefs[:10], chiefs[10:]
        params["person.login"] = ",".join(batch)
        result = []
        req = retry(session.get, [], dict(url=START_URL, params=params)).json()
        result.extend(req["result"])
        while "next" in req["links"]:
            req = retry(session.get, [], dict(url=req["links"]["next"])).json()
            result.extend(req["result"])
        for res in result:
            login = res["person"]["login"]
            if res.get("role") not in ("chief", "deputy", "general_director"):
                continue
            try:
                level = res["department_group"]["level"]
            except KeyError:
                continue
            if login not in chief_to_level or chief_to_level[login] > level:
                chief_to_level[login] = level

    with open(args.output, "w", encoding="utf8") as f:
        f.write("login\tlevel\n")
        for login in sorted(chief_to_level, key=lambda x: (chief_to_level[x], x)):
            f.write(f"{login}\t{chief_to_level[login]}\n")


if __name__ == "__main__":
    main()
