#!/usr/bin/env python
# -*- coding: utf-8 -*-
import os
from collections import Counter
import argparse
import yt.wrapper as yt

OUTPUT_TABLE_SCHEMA = [
    {"name": "id", "type": "uint64"},
    {"name": "login", "type": "string"},
    {"name": "yandex_login", "type": "string"},
    {"name": "chiefs", "type": "string"},
    {"name": "departments", "type": "string"},
    {"name": "department_grouping", "type": "string"},
    {"name": "position", "type": "string"},
    {"name": "position_grouping", "type": "string"},
    {"name": "management_level", "type": "uint64"},
]

DEPARTMENTS_LOWER = {
    "Команда поддержки бизнеса": "Команда поддержки бизнеса",
    "Буткемп мобильных разработчиков": "Буткемп",
    "Буткемп": None,
    "Браузер": "VS Суперапп",
    "Дизайн Яндекса и Дзена": None,
    "Отдел дата-журналистики и ML экспериментов": None,
    "Финансовые сервисы Яндекса": "Персональные и финансовые сервисы",
    "Персональные сервисы": "Персональные и финансовые сервисы",
    "Дивизион операционного маркетинга": None,
    "Рекламные технологии": "Управление рекламных продуктов",
    "Департамент технологий портала": None,
    "Поиск": None,
    "Умные устройства": None,
    "Направление Яндекс.Путешествия": None,
    "Направление международного развития Яндекс.Такси": None,
    "Направление Яндекс.Драйв": None,
    "Направление Яндекс.Еда": None,
    "Направление Яндекс.Лавка": None,
    "Подразделение Едадил": None,
    "Толока": None,
    "Практикум": None,
    "Департамент корпоративного развития": "Корпоративное развитие",
    "exЯндекс.Деньги": None,
    "RouteQ (Маршрутизация)": "RouteQ",
    "Образовательная инициатива": "Образование",
    "Поисковый портал": "Поиск",
    "Вертикали (Поддержка бизнеса)":"КПБ Вертикалей",
    "Эксплуатация": None,
    "Команда устойчивого развития": "Устойчивое развитие",
    "Департамент корпоративного развития": "Корпоративное развитие",
    "Внешние консультанты отдела исследований": "Команда поддержки бизнеса",
}
DEPARTMENTS_HIGHER = {
    "Беспилотные автомобили": None,
    "Вертикали": None,
    "Департамент корпоративных коммуникаций": None,
    "Департамент эксплуатации и строительства": None,
    "Дзен": None,
    "Доставка": None,
    "Коммерческий департамент": None,
    "Маркет": None,
    "Медиасервисы": None,
    "Облако": None,
    "Общие подразделения Екома и Райдтеха": None,
    "Операционный блок": None,
    "Отдел безопасности": None,
    "Райдтех": None,
    "Геосервисы": "Сервисы для малого бизнеса и Гео",
    "Услуги": None,
    "Финансы": "Финансовый департамент",
    "Фудтех": None,
    "Юристы": "Юридический департамент",
    "Яндекс.Практикум": None,
    "Управление персоналом": "HR-департамент",
}


def choose_deparment(rec):
    deps = set(rec["departments"].split(","))
    valid_lower_deps = deps & set(DEPARTMENTS_LOWER.keys())
    if valid_lower_deps:
        key = sorted(valid_lower_deps)[0]
        lower_dep = DEPARTMENTS_LOWER.get(key) or key
    else:
        lower_dep = None
    valid_higher_deps = deps & set(DEPARTMENTS_HIGHER.keys())
    if valid_higher_deps:
        key = sorted(valid_higher_deps)[0]
        higher_dep =  DEPARTMENTS_HIGHER.get(key) or key
    else:
        higher_dep = None
    if not higher_dep:
        if "Общие подразделения" in deps and "Бизнес-группа Eкома и Райдтеха" in deps:
            higher_dep = "Общие подразделения Екома и Райдтеха"
        else:
            higher_dep = "Другое"
    return ";".join(sorted([x for x in (lower_dep, higher_dep) if x]))


def choose_position(rec):
    pos = (rec["position"] or "").lower()
    if "менедж" in pos or "руководит" in pos:
        return "Менеджмент"
    if "аналит" in pos:
        return "Аналитика"
    if "разраб" in pos:
        return "Разработка"
    elif "тестир" in pos:
        return "Тестирование"
    elif "дизайн" in pos:
        return "Дизайн"
    else:
        return "Другое"


def read_tsv(filename):
    with open(filename) as f:
        cnt = f.read()
    cnt = [x for x in cnt.split("\n") if x]
    cnt = [x.split("\t") for x in cnt]
    return [dict(zip(cnt[0], x)) for x in cnt[1:]]


def wrap_1(x):
    if x == 1:
        return 2
    return x


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument("--input_staff", default="all_staff_data.tsv")
    parser.add_argument("--input_departments", default="department_staff.tsv")
    args = parser.parse_args()

    deps = read_tsv(args.input_departments)
    login_to_manager_level = {
        x["login"].lower(): wrap_1(int(x["level"])) for x in deps
    }
    del deps
    cnt = read_tsv(args.input_staff)

    result = []
    for rec in cnt:
        rec["id"] = int(rec["id"])
        rec["yandex_login"] = (rec.pop("yandex.login", "") or "").lower() or None
        rec["department_grouping"] = choose_deparment(rec)
        rec["position_grouping"] = choose_position(rec)
        rec["management_level"] = login_to_manager_level.get(rec["login"].lower()) or 100
        result.append(rec)
    yt.write_table(
        yt.TablePath(
            "//home/strm/vh_analytics/staff_data_private/staff_data",
            schema=OUTPUT_TABLE_SCHEMA,
        ),
        result,
    )


if __name__ == "__main__":
    main()
