#!/usr/bin/env python
# -*- coding: utf-8 -*-
import os
import argparse
import json
import time
import sys
from requests import Session
import tqdm

START_URL = "https://staff-api.yandex-team.ru/v3/persons"

def get_dep_name(dep):
    return (
        dep["name"]["short"]["ru"]
        or dep["name"]["full"]["ru"]
        or dep["name"]["short"]["en"]
        or dep["name"]["full"]["en"]
    )

def retry(method, args, kwargs):
    retries = 0
    while retries < 10:
        try:
            return method(*args, **kwargs)
        except ConnectionResetError:
            sys.stderr.write("encountered ConnectionResetError\n")
            retries += 1
            time.sleep(10)
    raise Exception("ConnectionResetError retries limit exceeded")


def make_line(res):
    id_ = res["id"]
    login = res["login"]
    try:
        yandex_login = res["yandex"]["login"] or ""
    except:
        yandex_login = ""
    position = res["official"]["position"]["ru"] or res["official"]["position"]["en"] or ""
    if res["chiefs"]:
        chiefs = ",".join([x["login"] for x in res["chiefs"]][::-1])
    else:
        chiefs = ""
    departments = []
    for a in res["department_group"]["ancestors"]:
        departments.append(get_dep_name(a["department"]))
    departments.append(get_dep_name(res["department_group"]["department"]))
    departments = ",".join(departments)
    return f"{id_}\t{login}\t{yandex_login}\t{position}\t{chiefs}\t{departments}\n"


def main():

    session = Session()
    session.headers.update({
        "Authorization": f"OAuth {os.environ['STAFF_TOKEN']}"
    })

    params = {
        "_nopage": 1,
        "_sort": "id",
        "_fields": "official.position,login,id,name,department_group,yandex.login,personal_emails,chiefs.login",
        "official.is_robot": 'false'
    }

    def all_logins(url=START_URL):
        while True:
            try:
                response = retry(session.get, [], dict(url=url, params=params))
                response = response.json()["result"]
            except KeyError:
                raise Exception(f"unintended response: {response.content.decode('utf8', errors='replace')}")
            if response:
                for res in response:
                    yield res
                params["_query"] = f"id>{response[-1]['id']}"
            else:
                break

    f = open("all_staff_data.tsv", "w", encoding="utf8")
    f.write("id\tlogin\tyandex.login\tposition\tchiefs\tdepartments\n")

    for res in tqdm.tqdm(all_logins()):
        f.write(make_line(res))
        del res



if __name__ == "__main__":
    main()
